/*
 * Copyright 2010-2026 Hyland Software, Inc. and its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.activiti.core.common.spring.security;

import java.security.Principal;
import java.util.Optional;
import org.activiti.api.runtime.shared.security.PrincipalIdentityProvider;
import org.springframework.security.core.Authentication;

public class AuthenticationPrincipalIdentityProvider implements PrincipalIdentityProvider {

    private static final String EMPTY_ANONYMOUS_USER_ID = "";

    @Override
    public String getUserId(Principal principal) {
        return Optional.of(principal)
            .filter(Authentication.class::isInstance)
            .map(Authentication.class::cast)
            .map(this::getUserId)
            .orElseThrow(this::securityException);
    }

    protected String getUserId(Authentication authentication) {
        return Optional.ofNullable(authentication.getName()).orElseGet(this::getAnonymousUserId);
    }

    protected String getAnonymousUserId() {
        return EMPTY_ANONYMOUS_USER_ID;
    }

    protected SecurityException securityException() {
        return new SecurityException("Invalid principal authentication object instance");
    }
}
