/*
 * Copyright 2017-2026 Hyland Software, Inc. and its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.activiti.cloud.security.feign.configuration;

import org.activiti.cloud.security.feign.ClientCredentialsAuthRequestInterceptor;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.security.oauth2.client.AuthorizedClientServiceOAuth2AuthorizedClientManager;
import org.springframework.security.oauth2.client.OAuth2AuthorizedClientProvider;
import org.springframework.security.oauth2.client.OAuth2AuthorizedClientProviderBuilder;
import org.springframework.security.oauth2.client.OAuth2AuthorizedClientService;
import org.springframework.security.oauth2.client.registration.ClientRegistration;
import org.springframework.security.oauth2.client.registration.ClientRegistrationRepository;

public class ClientCredentialsAuthConfiguration {

    public ClientCredentialsAuthRequestInterceptor clientCredentialsAuthRequestInterceptor(
        OAuth2AuthorizedClientService oAuth2AuthorizedClientService,
        ClientRegistrationRepository clientRegistrationRepository,
        ClientRegistration clientRegistration
    ) {
        return clientCredentialsAuthRequestInterceptor(
            oAuth2AuthorizedClientService,
            clientRegistrationRepository,
            clientRegistration,
            null
        );
    }

    public ClientCredentialsAuthRequestInterceptor clientCredentialsAuthRequestInterceptor(
        OAuth2AuthorizedClientService oAuth2AuthorizedClientService,
        ClientRegistrationRepository clientRegistrationRepository,
        ClientRegistration clientRegistration,
        OAuth2AuthorizedClientProvider provider
    ) {
        AuthorizedClientServiceOAuth2AuthorizedClientManager authorizedClientManager = new AuthorizedClientServiceOAuth2AuthorizedClientManager(
            clientRegistrationRepository,
            oAuth2AuthorizedClientService
        );

        OAuth2AuthorizedClientProviderBuilder oAuth2AuthorizedClientProviderBuilder = OAuth2AuthorizedClientProviderBuilder
            .builder()
            .refreshToken()
            .clientCredentials();
        OAuth2AuthorizedClientProvider authorizedClientProvider = provider == null
            ? oAuth2AuthorizedClientProviderBuilder.build()
            : oAuth2AuthorizedClientProviderBuilder.provider(provider).build();

        authorizedClientManager.setAuthorizedClientProvider(authorizedClientProvider);

        return new ClientCredentialsAuthRequestInterceptor(authorizedClientManager, clientRegistration);
    }

    public ClientRegistration clientRegistration(
        ClientRegistrationRepository clientRegistrationRepository,
        @Value("${activiti.cloud.services.oauth2.iam-name:keycloak}") String clientName
    ) {
        return clientRegistrationRepository.findByRegistrationId(clientName);
    }
}
