/*
 * Copyright 2017-2026 Hyland Software, Inc. and its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.activiti.services.connectors.channel;

import static org.springframework.transaction.annotation.Propagation.REQUIRES_NEW;

import org.activiti.cloud.api.process.model.impl.IntegrationErrorImpl;
import org.activiti.cloud.services.events.configuration.RuntimeBundleProperties;
import org.activiti.cloud.services.events.listeners.ProcessEngineEventsAggregator;
import org.activiti.engine.ManagementService;
import org.activiti.engine.impl.cmd.integration.DeleteIntegrationContextCmd;
import org.activiti.engine.impl.interceptor.Command;
import org.activiti.engine.impl.persistence.entity.integration.IntegrationContextEntity;
import org.springframework.transaction.annotation.Transactional;

public class ServiceTaskIntegrationCompletionHandler {

    private final RuntimeBundleProperties runtimeBundleProperties;
    private final ManagementService managementService;
    private final ProcessEngineEventsAggregator processEngineEventsAggregator;

    public ServiceTaskIntegrationCompletionHandler(
        RuntimeBundleProperties runtimeBundleProperties,
        ManagementService managementService,
        ProcessEngineEventsAggregator processEngineEventsAggregator
    ) {
        this.runtimeBundleProperties = runtimeBundleProperties;
        this.managementService = managementService;
        this.processEngineEventsAggregator = processEngineEventsAggregator;
    }

    @Transactional(propagation = REQUIRES_NEW)
    public void handlePropagationFailure(
        IntegrationErrorImpl integrationError,
        IntegrationContextEntity integrationContextEntity
    ) {
        Command<?> finalErrorHandlingCmd = CompositeCommand.of(
            new AggregateIntegrationErrorReceivedEventCmd(
                integrationError,
                runtimeBundleProperties,
                processEngineEventsAggregator
            ),
            new DeleteIntegrationContextCmd(integrationContextEntity)
        );

        managementService.executeCommand(finalErrorHandlingCmd);
    }
}
