/*
 * Decompiled with CFR 0.152.
 */
package org.alfresco.repo.security.authentication;

import com.fasterxml.uuid.Generators;
import java.io.Serializable;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Date;
import java.util.HashSet;
import java.util.Set;
import java.util.zip.CRC32;
import org.alfresco.repo.cache.SimpleCache;
import org.alfresco.repo.security.authentication.AuthenticationException;
import org.alfresco.repo.security.authentication.AuthenticationUtil;
import org.alfresco.repo.security.authentication.TicketComponent;
import org.alfresco.repo.security.authentication.TicketExpiredException;
import org.alfresco.service.cmr.repository.datatype.Duration;
import org.alfresco.util.GUID;
import org.alfresco.util.ParameterCheck;
import org.apache.commons.codec.binary.Hex;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class InMemoryTicketComponentImpl
implements TicketComponent {
    public static final String GRANTED_AUTHORITY_TICKET_PREFIX = "TICKET_";
    private static Log logger = LogFactory.getLog(InMemoryTicketComponentImpl.class);
    private static ThreadLocal<String> currentTicket = new ThreadLocal();
    private boolean ticketsExpire;
    private Duration validDuration;
    private boolean oneOff;
    private String guid;
    private SimpleCache<String, Ticket> ticketsCache;
    private SimpleCache<String, String> usernameToTicketIdCache;
    private ExpiryMode expiryMode = ExpiryMode.AFTER_INACTIVITY;
    private boolean useSingleTicketPerUser = true;

    public InMemoryTicketComponentImpl() {
        this.guid = GUID.generate();
    }

    public void setTicketsCache(SimpleCache<String, Ticket> ticketsCache) {
        this.ticketsCache = ticketsCache;
    }

    public void setUsernameToTicketIdCache(SimpleCache<String, String> usernameToTicketIdCache) {
        this.usernameToTicketIdCache = usernameToTicketIdCache;
    }

    public void setUseSingleTicketPerUser(boolean useSingleTicketPerUser) {
        this.useSingleTicketPerUser = useSingleTicketPerUser;
    }

    @Override
    public boolean getUseSingleTicketPerUser() {
        return this.useSingleTicketPerUser;
    }

    @Deprecated
    public void setOneOff(boolean oneOff) {
        this.oneOff = oneOff;
        if (this.oneOff) {
            logger.warn((Object)"The 'oneOff' feature has been deprecated and will be removed in a future version. This feature may not work as intended even in the current version");
        }
    }

    public void setTicketsExpire(boolean ticketsExpire) {
        this.ticketsExpire = ticketsExpire;
    }

    public void setExpiryMode(String expiryMode) {
        this.expiryMode = ExpiryMode.valueOf(expiryMode);
    }

    public void setValidDuration(String validDuration) {
        this.validDuration = new Duration(validDuration);
    }

    private void putIntoCache(Ticket ticket) {
        if (logger.isTraceEnabled()) {
            logger.trace((Object)("Putting into ticketsCache " + this.ticketsCache.toString() + " ticket: " + String.valueOf(ticket)));
        }
        this.ticketsCache.put(ticket.getTicketId(), ticket);
        if (logger.isTraceEnabled()) {
            logger.trace((Object)("Putting into usernameToTicketIdCache " + this.usernameToTicketIdCache.toString() + " username and ticketId of: " + String.valueOf(ticket)));
        }
        this.usernameToTicketIdCache.put(ticket.getUserName(), ticket.getTicketId());
    }

    private void removeFromCache(String ticketId) {
        String username;
        String actualUserTicketIdFromCache;
        Ticket ticket = null;
        if (ticketId != null) {
            ticket = this.ticketsCache.get(ticketId);
        }
        if (logger.isTraceEnabled()) {
            logger.trace((Object)("Removing ticket from ticketsCache: " + ticketId));
        }
        this.ticketsCache.remove(ticketId);
        if (ticket != null && ticketId.equals(actualUserTicketIdFromCache = this.usernameToTicketIdCache.get(username = ticket.getUserName()))) {
            if (logger.isTraceEnabled()) {
                logger.trace((Object)("Removing ticketId from usernameToTicketIdCache for: " + username));
            }
            this.usernameToTicketIdCache.remove(username);
        }
    }

    @Override
    public String getNewTicket(String userName) throws AuthenticationException {
        Ticket ticket = null;
        if (this.useSingleTicketPerUser) {
            ticket = this.findNonExpiredUserTicket(userName);
        }
        if (ticket == null) {
            Date expiryDate = null;
            if (this.ticketsExpire) {
                expiryDate = Duration.add(new Date(), this.validDuration);
            }
            ticket = new Ticket(this.ticketsExpire ? this.expiryMode : ExpiryMode.DO_NOT_EXPIRE, expiryDate, userName, this.validDuration);
            this.putIntoCache(ticket);
        }
        String ticketString = GRANTED_AUTHORITY_TICKET_PREFIX + ticket.getTicketId();
        currentTicket.set(ticketString);
        if (logger.isTraceEnabled()) {
            logger.trace((Object)("Setting the current ticket for this thread: " + Thread.currentThread().getName() + " to: " + ticketString));
        }
        return ticketString;
    }

    private Ticket findNonExpiredUserTicket(String userName) {
        Ticket newTicket;
        Ticket ticketFromCache;
        String userTicketIdFromCache = this.usernameToTicketIdCache.get(userName);
        if (userTicketIdFromCache != null && (ticketFromCache = this.ticketsCache.get(userTicketIdFromCache)) != null && (newTicket = ticketFromCache.getNewEntry()) != null) {
            if (newTicket != ticketFromCache) {
                this.putIntoCache(newTicket);
            }
            return newTicket;
        }
        return null;
    }

    @Override
    public String validateTicket(String ticketString) throws AuthenticationException {
        String ticketKey;
        Ticket ticket;
        if (logger.isTraceEnabled()) {
            logger.trace((Object)("Validating ticket: " + ticketString));
        }
        if ((ticket = this.ticketsCache.get(ticketKey = this.getTicketKey(ticketString))) == null) {
            String msg = "Missing ticket for " + ticketString;
            if (logger.isDebugEnabled()) {
                logger.debug((Object)msg);
            }
            throw new AuthenticationException(msg);
        }
        Ticket newTicket = ticket.getNewEntry();
        if (newTicket == null) {
            String msg = "Ticket expired for " + ticketString;
            if (logger.isDebugEnabled()) {
                logger.debug((Object)msg);
            }
            throw new TicketExpiredException(msg);
        }
        if (this.oneOff) {
            this.removeFromCache(ticketKey);
        } else if (newTicket != ticket) {
            this.putIntoCache(newTicket);
        }
        currentTicket.set(ticketString);
        if (logger.isTraceEnabled()) {
            logger.trace((Object)("Setting the current ticket for this thread: " + Thread.currentThread().getName() + " to: " + ticketString));
        }
        return newTicket.getUserName();
    }

    private Ticket getTicketByTicketString(String ticketString) {
        Ticket ticket = this.ticketsCache.get(this.getTicketKey(ticketString));
        return ticket;
    }

    private String getTicketKey(String ticketString) {
        if (ticketString == null) {
            return null;
        }
        if (ticketString.length() < GRANTED_AUTHORITY_TICKET_PREFIX.length()) {
            throw new AuthenticationException(ticketString + " is an invalid ticket format");
        }
        String key = ticketString.substring(GRANTED_AUTHORITY_TICKET_PREFIX.length());
        return key;
    }

    @Override
    public void invalidateTicketById(String ticketString) {
        String key = ticketString.substring(GRANTED_AUTHORITY_TICKET_PREFIX.length());
        this.removeFromCache(key);
    }

    @Override
    public Set<String> getUsersWithTickets(boolean nonExpiredOnly) {
        Date now = new Date();
        HashSet<String> users = new HashSet<String>();
        for (String key : this.ticketsCache.getKeys()) {
            Ticket ticket = this.ticketsCache.get(key);
            if (ticket == null || nonExpiredOnly && ticket.hasExpired(now)) continue;
            users.add(ticket.getUserName());
        }
        return users;
    }

    @Override
    public int countTickets(boolean nonExpiredOnly) {
        Date now = new Date();
        if (nonExpiredOnly) {
            int count = 0;
            for (String key : this.ticketsCache.getKeys()) {
                Ticket ticket = this.ticketsCache.get(key);
                if (ticket == null || ticket.hasExpired(now)) continue;
                ++count;
            }
            return count;
        }
        return this.ticketsCache.getKeys().size();
    }

    @Override
    public int invalidateTickets(boolean expiredOnly) {
        if (logger.isTraceEnabled()) {
            logger.trace((Object)("Invalidate all tickets, expired only: " + expiredOnly));
        }
        Date now = new Date();
        int count = 0;
        if (!expiredOnly) {
            count = this.ticketsCache.getKeys().size();
            if (logger.isTraceEnabled()) {
                logger.trace((Object)("Clearing all tickets from the ticketsCache, that used to have size: " + count));
            }
            this.ticketsCache.clear();
            this.usernameToTicketIdCache.clear();
        } else {
            HashSet<String> toRemove = new HashSet<String>();
            for (String key : this.ticketsCache.getKeys()) {
                Ticket ticket = this.ticketsCache.get(key);
                if (ticket != null && !ticket.hasExpired(now)) continue;
                ++count;
                toRemove.add(key);
            }
            for (String id : toRemove) {
                this.removeFromCache(id);
            }
        }
        return count;
    }

    @Override
    public void invalidateTicketByUser(String userName) {
        HashSet<String> toRemove = new HashSet<String>();
        for (String key : this.ticketsCache.getKeys()) {
            Ticket ticket = this.ticketsCache.get(key);
            if (ticket == null || !ticket.getUserName().equals(userName)) continue;
            toRemove.add(ticket.getTicketId());
        }
        for (String id : toRemove) {
            this.removeFromCache(id);
        }
    }

    public int hashCode() {
        int PRIME = 31;
        int result = 1;
        result = 31 * result + (this.guid == null ? 0 : this.guid.hashCode());
        return result;
    }

    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (this.getClass() != obj.getClass()) {
            return false;
        }
        InMemoryTicketComponentImpl other = (InMemoryTicketComponentImpl)obj;
        return !(this.guid == null ? other.guid != null : !this.guid.equals(other.guid));
    }

    @Override
    public String getAuthorityForTicket(String ticketString) {
        Ticket ticket = this.getTicketByTicketString(ticketString);
        if (ticket == null) {
            return null;
        }
        return ticket.getUserName();
    }

    @Override
    public String getCurrentTicket(String userName, boolean autoCreate) {
        String ticket = currentTicket.get();
        if (ticket == null) {
            return autoCreate ? this.getNewTicket(userName) : null;
        }
        String ticketUser = this.getAuthorityForTicket(ticket);
        if (userName.equals(ticketUser)) {
            return ticket;
        }
        return autoCreate ? this.getNewTicket(userName) : null;
    }

    @Override
    public void clearCurrentTicket() {
        InMemoryTicketComponentImpl.clearCurrentSecurityContext();
    }

    public static void clearCurrentSecurityContext() {
        String prevTicket = currentTicket.get();
        currentTicket.set(null);
        if (logger.isTraceEnabled()) {
            logger.trace((Object)("Clearing the current ticket for this thread: " + Thread.currentThread().getName() + " . Previous ticket was: " + prevTicket));
        }
    }

    public static enum ExpiryMode {
        AFTER_INACTIVITY,
        AFTER_FIXED_TIME,
        DO_NOT_EXPIRE;

    }

    public static class Ticket
    implements Serializable {
        private static final long serialVersionUID = -5904510560161261049L;
        private final ExpiryMode expires;
        private final Date expiryDate;
        private final String userName;
        private final String ticketId;
        private final Duration validDuration;
        private final Duration testDuration;

        Ticket(ExpiryMode expires, Date expiryDate, String userName, Duration validDuration) {
            Ticket.checkValidTicketParameters(expires, expiryDate, userName, validDuration);
            this.expires = expires;
            this.expiryDate = expiryDate;
            this.userName = userName;
            this.validDuration = validDuration;
            this.testDuration = validDuration.divide(2);
            String guid = Generators.randomBasedGenerator().generate().toString();
            this.ticketId = Ticket.computeTicketId(expires, expiryDate, userName, guid);
            if (logger.isTraceEnabled()) {
                logger.trace((Object)("Creating new ticket for user: " + AuthenticationUtil.maskUsername(userName) + " with ticketId: " + this.ticketId));
            }
        }

        private static String computeTicketId(ExpiryMode expires, Date expiryDate, String userName, String guid) {
            String ticketId;
            String encode = expires.toString() + Ticket.getExpireDateAsString(expiryDate) + userName + guid;
            try {
                MessageDigest digester = MessageDigest.getInstance("SHA-1");
                ticketId = new String(Hex.encodeHex((byte[])digester.digest(encode.getBytes())));
            }
            catch (NoSuchAlgorithmException e) {
                try {
                    MessageDigest digester = MessageDigest.getInstance("MD5");
                    ticketId = new String(Hex.encodeHex((byte[])digester.digest(encode.getBytes())));
                }
                catch (NoSuchAlgorithmException e1) {
                    CRC32 crc = new CRC32();
                    crc.update(encode.getBytes());
                    byte[] bytes = new byte[4];
                    long value = crc.getValue();
                    bytes[0] = (byte)(value & 0xFFL);
                    bytes[1] = (byte)((value >>>= 4) & 0xFFL);
                    bytes[2] = (byte)((value >>>= 4) & 0xFFL);
                    bytes[3] = (byte)((value >>>= 4) & 0xFFL);
                    ticketId = new String(Hex.encodeHex((byte[])bytes));
                }
            }
            return ticketId;
        }

        private Ticket(ExpiryMode expires, Date expiryDate, String userName, Duration validDuration, String ticketId) {
            Ticket.checkValidTicketParameters(expires, expiryDate, userName, validDuration);
            ParameterCheck.mandatory((String)"ticketId", (Object)ticketId);
            this.expires = expires;
            this.expiryDate = expiryDate;
            this.userName = userName;
            this.validDuration = validDuration;
            Duration tenPercent = validDuration.divide(10);
            this.testDuration = validDuration.subtract(tenPercent);
            this.ticketId = ticketId;
            if (logger.isTraceEnabled()) {
                logger.trace((Object)("Creating (cloning) new ticket for user: " + AuthenticationUtil.maskUsername(userName) + " with ticketId: " + this.ticketId));
            }
        }

        private static void checkValidTicketParameters(ExpiryMode expires, Date expiryDate, String userName, Duration validDuration) {
            ParameterCheck.mandatory((String)"expires mode", (Object)((Object)expires));
            ParameterCheck.mandatoryString((String)"userName", (String)userName);
            ParameterCheck.mandatory((String)"validDuration", (Object)validDuration);
            if (!ExpiryMode.DO_NOT_EXPIRE.equals((Object)expires)) {
                ParameterCheck.mandatory((String)"expiryDate", (Object)expiryDate);
            }
        }

        private static String getExpireDateAsString(Date expiryDate) {
            return expiryDate == null ? "" : expiryDate.toString();
        }

        boolean hasExpired(Date now) {
            return this.expiryDate != null && this.expiryDate.compareTo(now) < 0;
        }

        Ticket getNewEntry() {
            switch (this.expires.ordinal()) {
                case 1: {
                    if (this.hasExpired(new Date())) {
                        return null;
                    }
                    return this;
                }
                case 0: {
                    Date now = new Date();
                    if (this.hasExpired(now)) {
                        return null;
                    }
                    Duration remaining = new Duration(now, this.expiryDate);
                    if (remaining.compareTo(this.testDuration) < 0) {
                        if (logger.isTraceEnabled()) {
                            logger.trace((Object)("AFTER_INACTIVITY case, Creating new ticket based on the current one that expires at: " + String.valueOf(this.expiryDate)));
                        }
                        return new Ticket(this.expires, Duration.add(now, this.validDuration), this.userName, this.validDuration, this.ticketId);
                    }
                    return this;
                }
            }
            return this;
        }

        public boolean equals(Object o) {
            if (o == this) {
                return true;
            }
            if (!(o instanceof Ticket)) {
                return false;
            }
            Ticket t = (Ticket)o;
            return this.ticketId.equals(t.ticketId) && this.userName.equals(t.userName) && this.expires.equals((Object)t.expires) && this.areTheExpiryDatesEqual(t);
        }

        private boolean areTheExpiryDatesEqual(Ticket t) {
            if (ExpiryMode.DO_NOT_EXPIRE.equals((Object)this.expires)) {
                return true;
            }
            if (this.expiryDate == null) {
                logger.warn((Object)"expiryDate should not be null in this case");
            }
            return this.expiryDate.equals(t.expiryDate);
        }

        public int hashCode() {
            return this.ticketId.hashCode();
        }

        protected ExpiryMode getExpires() {
            return this.expires;
        }

        protected Date getExpiryDate() {
            return this.expiryDate;
        }

        protected String getTicketId() {
            return this.ticketId;
        }

        protected String getUserName() {
            return this.userName;
        }

        public String toString() {
            return "Ticket with ticketId: " + this.ticketId + " for user: " + AuthenticationUtil.maskUsername(this.userName) + " ExpiryMode: " + String.valueOf((Object)this.expires) + " expire date: " + String.valueOf(this.expiryDate) + " validDuration: " + String.valueOf(this.validDuration);
        }
    }
}

