/*-
 * #%L
 * Alfresco JMX Logger
 * %%
 * Copyright (C) 2022 Alfresco Software Limited
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */
package org.alfresco.util.log.log4j2;

import javax.management.ObjectName;
import java.io.Serializable;
import java.lang.management.ManagementFactory;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

import jmxlogger.tools.JmxEventLogger;
import jmxlogger.tools.ToolBox;
import org.apache.logging.log4j.core.Appender;
import org.apache.logging.log4j.core.Core;
import org.apache.logging.log4j.core.Filter;
import org.apache.logging.log4j.core.Layout;
import org.apache.logging.log4j.core.LogEvent;
import org.apache.logging.log4j.core.appender.AbstractAppender;
import org.apache.logging.log4j.core.config.Configuration;
import org.apache.logging.log4j.core.config.plugins.Plugin;
import org.apache.logging.log4j.core.config.plugins.PluginAttribute;
import org.apache.logging.log4j.core.config.plugins.PluginConfiguration;
import org.apache.logging.log4j.core.config.plugins.PluginElement;
import org.apache.logging.log4j.core.config.plugins.PluginFactory;

/**
 * A custom Log4J2 appender that pushes received log events to JMX.
 *
 * @author Domenico Sibilio
 */
@Plugin(name = "Log4j2JmxLogAppender", category = Core.CATEGORY_NAME, elementType = Appender.ELEMENT_TYPE)
public class Log4j2JmxLogAppender extends AbstractAppender
{
    private static final Map<String, String> LOG_CTX_BY_OBJECT_NAME = new HashMap<>();
    private final String serverSelection;
    private JmxEventLogger logger;

    /**
     * The factory method within this class should be used instead.
     *
     * @param name             the name of the appender.
     * @param filter           the filter to apply to incoming log events.
     * @param layout           the layout to apply to the log message.
     * @param objectName       the ObjectName for the underlying JMX bean.
     * @param serverSelection  the MBean Server for the underlying JMX bean.
     * @param ignoreExceptions whether this appender should ignore exceptions or propagate them.
     * @param configuration    the Log4j configuration.
     */
    public Log4j2JmxLogAppender(
        String name,
        Filter filter,
        Layout<? extends Serializable> layout,
        String objectName,
        String serverSelection,
        boolean ignoreExceptions,
        Configuration configuration)
    {
        super(name, filter, layout, ignoreExceptions, null);
        this.serverSelection = serverSelection;
        Objects.requireNonNull(serverSelection, "serverSelection should not be null");
        Objects.requireNonNull(configuration, "configuration should not be null");

        String logContextName = configuration.getLoggerContext().getName();
        ObjectName actualObjectName = buildObjectName(objectName, logContextName);
        setLogger(JmxEventLogger.createInstance());
        getLogger().setObjectName(actualObjectName);
        configureLogger();
        startLogger();
        LOG_CTX_BY_OBJECT_NAME.put(actualObjectName.getCanonicalName(), logContextName);
    }

    /**
     * Factory method to create a {@link Log4j2JmxLogAppender} instance.
     *
     * @param name             the name of the appender.
     * @param filter           the filter to apply to incoming log events.
     * @param layout           the layout to apply to the log message.
     * @param objectName       the ObjectName for the underlying JMX bean.
     * @param serverSelection  the MBean Server for the underlying JMX bean.
     * @param ignoreExceptions whether this appender should ignore exceptions or propagate them.
     * @return the newly built {@link Log4j2JmxLogAppender} instance.
     */
    @SuppressWarnings("unused")
    @PluginFactory
    public static Log4j2JmxLogAppender createAppender(
        @PluginAttribute("name") String name,
        @PluginElement("Filter") Filter filter,
        @PluginElement("Layout") Layout<? extends Serializable> layout,
        @PluginAttribute("ObjectName") String objectName,
        @PluginAttribute(value = "serverSelection", defaultString = "platform") String serverSelection,
        @PluginAttribute(value = "ignoreExceptions", defaultBoolean = true) boolean ignoreExceptions,
        @PluginConfiguration Configuration configuration)
    {
        return new Log4j2JmxLogAppender(name, filter, layout, objectName, serverSelection, ignoreExceptions,
            configuration);
    }

    @Override
    public void append(LogEvent event)
    {
        try
        {
            String formattedMessage = new String(getLayout().toByteArray(event));
            getLogger().log(prepareLogEvent(formattedMessage, event));
        }
        catch (Exception e)
        {
            if (!ignoreExceptions())
            {
                throw e;
            }

            error("Unable to send log to JMX.", e);
        }
    }

    /**
     * Convert the specified LogEvent and its formatted message into {@link Map} form.
     *
     * @param formattedMessage the log message formatted according to the specified layout.
     * @param logEvent         the log event with all the attached information.
     * @return a map rendition of the input logEvent.
     */
    private Map<String, Object> prepareLogEvent(String formattedMessage, LogEvent logEvent)
    {
        Map<String, Object> event = new HashMap<>();
        event.put(ToolBox.KEY_EVENT_SOURCE, this.getClass().getName());
        event.put(ToolBox.KEY_EVENT_LEVEN, logEvent.getLevel().toString());
        event.put(ToolBox.KEY_EVENT_LOGGER, logEvent.getLoggerName());
        event.put(ToolBox.KEY_EVENT_MESSAGE, formattedMessage);
        event.put(ToolBox.KEY_EVENT_SEQ_NUM, logEvent.getTimeMillis());
        event.put(ToolBox.KEY_EVENT_SOURCE_CLASS, logEvent.getLoggerFqcn());
        event.put(ToolBox.KEY_EVENT_SOURCE_METHOD, "Unavailable");
        event.put(ToolBox.KEY_EVENT_THREAD, logEvent.getThreadName());
        event.put(ToolBox.KEY_EVENT_THROWABLE, logEvent.getThrown());
        event.put(ToolBox.KEY_EVENT_TIME_STAMP, logEvent.getTimeMillis());
        return event;
    }

    /**
     * Configure the underlying {@link JmxEventLogger}.
     */
    private void configureLogger()
    {
        if (getLogger().getMBeanServer() == null)
        {
            if (getServerSelection().equalsIgnoreCase("platform"))
            {
                getLogger().setMBeanServer(ManagementFactory.getPlatformMBeanServer());
            }
            else
            {
                getLogger().setMBeanServer(ToolBox.findMBeanServer(getServerSelection()));
            }
        }
    }

    /**
     * Start the underlying {@link JmxEventLogger}.
     */
    private void startLogger()
    {
        if (!getLogger().isStarted())
        {
            getLogger().start();
        }
    }

    /**
     * Defaults to building a random object name in case of null input or
     * already-existing object name that is not part of the same Log4j2 context.
     *
     * @param objectName     the objectName to build.
     * @param logContextName the Log4j2 context name.
     * @return the built ObjectName.
     */
    private ObjectName buildObjectName(String objectName, String logContextName)
    {
        return objectName == null || (LOG_CTX_BY_OBJECT_NAME.get(objectName) != null &&
            !LOG_CTX_BY_OBJECT_NAME.get(objectName).equals(logContextName)) ?
            ToolBox.buildDefaultObjectName(Integer.toString(this.hashCode())) :
            ToolBox.buildObjectName(objectName);
    }

    public JmxEventLogger getLogger()
    {
        return logger;
    }

    public void setLogger(JmxEventLogger logger)
    {
        this.logger = logger;
    }

    public String getServerSelection()
    {
        return serverSelection;
    }
}
