/*
 * Decompiled with CFR 0.152.
 */
package org.alfresco.repo.content.caching.cleanup;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.PrintWriter;
import java.time.Duration;
import java.util.Calendar;
import java.util.GregorianCalendar;
import org.alfresco.repo.content.caching.CacheFileProps;
import org.alfresco.repo.content.caching.CachingContentStore;
import org.alfresco.repo.content.caching.ContentCacheImpl;
import org.alfresco.repo.content.caching.Key;
import org.alfresco.repo.content.caching.cleanup.CachedContentCleaner;
import org.alfresco.service.cmr.repository.ContentReader;
import org.alfresco.util.ApplicationContextHelper;
import org.alfresco.util.GUID;
import org.alfresco.util.testing.category.LuceneTests;
import org.apache.commons.io.FileUtils;
import org.awaitility.Awaitility;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.springframework.context.ApplicationContext;

@Category(value={LuceneTests.class})
public class CachedContentCleanupJobTest {
    private static final Duration MAX_WAIT_TIMEOUT = Duration.ofSeconds(10L);
    private static ApplicationContext ctx;
    private CachingContentStore cachingStore;
    private ContentCacheImpl cache;
    private File cacheRoot;
    private CachedContentCleaner cleaner;

    @BeforeClass
    public static void beforeClass() {
        String cleanerConf = "classpath:cachingstore/test-cleaner-context.xml";
        ctx = ApplicationContextHelper.getApplicationContext((String[])new String[]{cleanerConf});
    }

    @Before
    public void setUp() throws IOException {
        this.cachingStore = (CachingContentStore)ctx.getBean("cachingContentStore");
        this.cache = (ContentCacheImpl)ctx.getBean("contentCache");
        this.cacheRoot = this.cache.getCacheRoot();
        this.cleaner = (CachedContentCleaner)ctx.getBean("cachedContentCleaner");
        this.cleaner.setMinFileAgeMillis(0L);
        this.cleaner.setMaxDeleteWatchCount(Integer.valueOf(0));
        this.cache.removeAll();
        FileUtils.cleanDirectory((File)this.cacheRoot);
    }

    @Test
    public void filesNotInCacheAreDeleted() throws InterruptedException {
        this.cleaner.setMaxDeleteWatchCount(Integer.valueOf(0));
        int numFiles = 300;
        long totalSize = 0L;
        File[] files = new File[numFiles];
        int i = 0;
        while (i < numFiles) {
            File cacheFile;
            UrlSource urlSource = UrlSource.values()[i % UrlSource.values().length];
            files[i] = cacheFile = this.createCacheFile(urlSource, false);
            totalSize += cacheFile.length();
            ++i;
        }
        this.cleaner.execute();
        Awaitility.await().pollDelay(Duration.ofMillis(100L)).atMost(MAX_WAIT_TIMEOUT).until(() -> !this.cleaner.isRunning());
        File[] fileArray = files;
        int n = files.length;
        int n2 = 0;
        while (n2 < n) {
            File file = fileArray[n2];
            Assert.assertFalse((String)("File should have been deleted: " + String.valueOf(file)), (boolean)file.exists());
            ++n2;
        }
        Assert.assertEquals((String)"Incorrect number of deleted files", (long)numFiles, (long)this.cleaner.getNumFilesDeleted());
        Assert.assertEquals((String)"Incorrect total size of files deleted", (long)totalSize, (long)this.cleaner.getSizeFilesDeleted());
    }

    @Test
    public void filesNewerThanMinFileAgeMillisAreNotDeleted() throws InterruptedException {
        this.cleaner.setMinFileAgeMillis(5000L);
        this.cleaner.setMaxDeleteWatchCount(Integer.valueOf(0));
        int numFiles = 10;
        File[] oldFiles = new File[numFiles];
        int i = 0;
        while (i < numFiles) {
            oldFiles[i] = this.createCacheFile(UrlSource.REVERSE_CACHE_LOOKUP, false);
            ++i;
        }
        Thread.sleep(5000L);
        File[] newFiles = new File[numFiles];
        long newFilesTotalSize = 0L;
        int i2 = 0;
        while (i2 < numFiles) {
            newFiles[i2] = this.createCacheFile(UrlSource.REVERSE_CACHE_LOOKUP, false);
            newFilesTotalSize += newFiles[i2].length();
            ++i2;
        }
        this.cleaner.execute();
        Awaitility.await().pollDelay(Duration.ofMillis(100L)).atMost(MAX_WAIT_TIMEOUT).until(() -> !this.cleaner.isRunning());
        if (this.cleaner.getDurationMillis() > 5000L) {
            Assert.fail((String)("Test unable to complete, since cleaner took " + this.cleaner.getDurationMillis() + "ms which is longer than minFileAge [5000ms]"));
        }
        File[] fileArray = oldFiles;
        int n = oldFiles.length;
        int n2 = 0;
        while (n2 < n) {
            File file = fileArray[n2];
            Assert.assertFalse((String)("File should have been deleted: " + String.valueOf(file)), (boolean)file.exists());
            ++n2;
        }
        fileArray = newFiles;
        n = newFiles.length;
        n2 = 0;
        while (n2 < n) {
            File file = fileArray[n2];
            Assert.assertTrue((String)("File should not have been deleted: " + String.valueOf(file)), (boolean)file.exists());
            ++n2;
        }
        Assert.assertEquals((String)"Incorrect number of deleted files", (long)newFiles.length, (long)this.cleaner.getNumFilesDeleted());
        Assert.assertEquals((String)"Incorrect total size of files deleted", (long)newFilesTotalSize, (long)this.cleaner.getSizeFilesDeleted());
    }

    @Test
    public void aggressiveCleanReclaimsTargetSpace() throws InterruptedException {
        int numFiles = 30;
        File[] files = new File[numFiles];
        int i = 0;
        while (i < numFiles) {
            files[i] = this.createCacheFile(UrlSource.REVERSE_CACHE_LOOKUP, true);
            ++i;
        }
        long fileSize = files[0].length();
        long sevenFilesSize = 7L * fileSize;
        this.cleaner.executeAggressive("aggressiveCleanReclaimsTargetSpace()", sevenFilesSize);
        Thread.sleep(400L);
        while (this.cleaner.isRunning()) {
            Thread.sleep(200L);
        }
        int numDeleted = 0;
        File[] fileArray = files;
        int n = files.length;
        int n2 = 0;
        while (n2 < n) {
            File f = fileArray[n2];
            if (!f.exists()) {
                ++numDeleted;
            }
            ++n2;
        }
        Assert.assertEquals((String)"Wrong number of files deleted", (long)7L, (long)numDeleted);
        Assert.assertEquals((String)"Incorrect number of deleted files", (long)7L, (long)this.cleaner.getNumFilesDeleted());
        Assert.assertEquals((String)"Incorrect total size of files deleted", (long)sevenFilesSize, (long)this.cleaner.getSizeFilesDeleted());
    }

    @Test
    public void standardCleanAfterAggressiveFinished() throws InterruptedException {
        File[] files = new File[30];
        int i = 0;
        while (i < 30) {
            GregorianCalendar calendar = new GregorianCalendar(2010, 11, 2, 17, i);
            files[i] = i >= 21 && i <= 24 ? this.createCacheFile(calendar, UrlSource.NOT_PRESENT, false) : this.createCacheFile(calendar, UrlSource.REVERSE_CACHE_LOOKUP, true);
            ++i;
        }
        long fileSize = files[0].length();
        long sevenFilesSize = 7L * fileSize;
        this.cleaner.executeAggressive("standardCleanAfterAggressiveFinished()", sevenFilesSize);
        Thread.sleep(400L);
        while (this.cleaner.isRunning()) {
            Thread.sleep(200L);
        }
        int i2 = 0;
        while (i2 < 30) {
            if (i2 < 7) {
                Assert.assertFalse((String)"First 7 files should have been aggressively cleaned", (boolean)files[i2].exists());
            }
            if (i2 >= 21 && i2 <= 24) {
                Assert.assertFalse((String)"Files with indexes 21-24 should have been deleted", (boolean)files[i2].exists());
            }
            ++i2;
        }
        Assert.assertEquals((String)"Incorrect number of deleted files", (long)11L, (long)this.cleaner.getNumFilesDeleted());
        Assert.assertEquals((String)"Incorrect total size of files deleted", (long)(11L * fileSize), (long)this.cleaner.getSizeFilesDeleted());
    }

    @Test
    public void emptyParentDirectoriesAreDeleted() throws FileNotFoundException {
        this.cleaner.setMaxDeleteWatchCount(Integer.valueOf(0));
        File file = new File(this.cacheRoot, "243235984/a/b/c/d.bin");
        file.getParentFile().mkdirs();
        PrintWriter writer = new PrintWriter(file);
        writer.println("Content for emptyParentDirectoriesAreDeleted");
        writer.close();
        Assert.assertTrue((String)"Directory should exist", (boolean)new File(this.cacheRoot, "243235984/a/b/c").exists());
        this.cleaner.handle(file);
        Assert.assertFalse((String)"Directory should have been deleted", (boolean)new File(this.cacheRoot, "243235984").exists());
    }

    @Test
    public void markedFilesHaveDeletionDeferredUntilCorrectPassOfCleaner() {
        this.cleaner.setMaxDeleteWatchCount(Integer.valueOf(0));
        File file = this.createCacheFile(UrlSource.NOT_PRESENT, false);
        this.cleaner.handle(file);
        this.checkFilesDeleted(file);
        this.cleaner.setMaxDeleteWatchCount(Integer.valueOf(1));
        file = this.createCacheFile(UrlSource.NOT_PRESENT, false);
        this.cleaner.handle(file);
        this.checkWatchCountForCacheFile(file, 1);
        this.cleaner.handle(file);
        this.checkFilesDeleted(file);
        this.cleaner.setMaxDeleteWatchCount(Integer.valueOf(3));
        file = this.createCacheFile(UrlSource.NOT_PRESENT, false);
        this.cleaner.handle(file);
        this.checkWatchCountForCacheFile(file, 1);
        this.cleaner.handle(file);
        this.checkWatchCountForCacheFile(file, 2);
        this.cleaner.handle(file);
        this.checkWatchCountForCacheFile(file, 3);
        this.cleaner.handle(file);
        this.checkFilesDeleted(file);
    }

    private void checkFilesDeleted(File file) {
        Assert.assertFalse((String)("File should have been deleted: " + String.valueOf(file)), (boolean)file.exists());
        CacheFileProps props = new CacheFileProps(file);
        Assert.assertFalse((String)("Properties file should have been deleted, cache file: " + String.valueOf(file)), (boolean)props.exists());
    }

    private void checkWatchCountForCacheFile(File file, Integer expectedWatchCount) {
        Assert.assertTrue((String)("File should still exist: " + String.valueOf(file)), (boolean)file.exists());
        CacheFileProps props = new CacheFileProps(file);
        props.load();
        Assert.assertEquals((String)"File should contain correct deleteWatchCount", (Object)expectedWatchCount, (Object)props.getDeleteWatchCount());
    }

    @Test
    public void filesInCacheAreNotDeleted() throws InterruptedException {
        this.cleaner.setMaxDeleteWatchCount(Integer.valueOf(0));
        String url = this.makeContentUrl();
        int numFiles = 50;
        int i = 0;
        while (i < numFiles) {
            ContentReader reader = this.cachingStore.getReader(url);
            reader.getContentString();
            ++i;
        }
        this.cleaner.execute();
        Thread.sleep(400L);
        while (this.cleaner.isRunning()) {
            Thread.sleep(200L);
        }
        i = 0;
        while (i < numFiles) {
            File cacheFile = new File(this.cache.getCacheFilePath(url));
            Assert.assertTrue((String)"File should exist", (boolean)cacheFile.exists());
            ++i;
        }
    }

    private File createCacheFile(UrlSource urlSource, boolean putInCache) {
        GregorianCalendar calendar = new GregorianCalendar();
        return this.createCacheFile(calendar, urlSource, putInCache);
    }

    private File createCacheFile(Calendar calendar, UrlSource urlSource, boolean putInCache) {
        File file = new File(this.cacheRoot, this.createNewCacheFilePath(calendar));
        file.getParentFile().mkdirs();
        this.writeSampleContent(file);
        String contentUrl = this.makeContentUrl();
        if (putInCache) {
            this.cache.putIntoLookup(Key.forUrl((String)contentUrl), file.getAbsolutePath());
        }
        switch (urlSource) {
            case NOT_PRESENT: {
                break;
            }
            case PROPS_FILE: {
                CacheFileProps props = new CacheFileProps(file);
                props.setContentUrl(contentUrl);
                props.store();
                break;
            }
            case REVERSE_CACHE_LOOKUP: {
                this.cache.putIntoLookup(Key.forCacheFile((File)file), contentUrl);
            }
        }
        Assert.assertTrue((String)"File should exist", (boolean)file.exists());
        return file;
    }

    private String createNewCacheFilePath(Calendar calendar) {
        int year = calendar.get(1);
        int month = calendar.get(2) + 1;
        int day = calendar.get(5);
        int hour = calendar.get(11);
        int minute = calendar.get(12);
        StringBuilder sb = new StringBuilder(20);
        sb.append(year).append('/').append(month).append('/').append(day).append('/').append(hour).append('/').append(minute).append('/').append(GUID.generate()).append(".bin");
        return sb.toString();
    }

    private String makeContentUrl() {
        return "protocol://some/made/up/url/" + GUID.generate();
    }

    private void writeSampleContent(File file) {
        try {
            PrintWriter writer = new PrintWriter(file);
            writer.println("Content for sample file in " + this.getClass().getName());
            writer.close();
        }
        catch (Throwable e) {
            throw new RuntimeException("Couldn't write file: " + String.valueOf(file), e);
        }
    }

    private static enum UrlSource {
        PROPS_FILE,
        REVERSE_CACHE_LOOKUP,
        NOT_PRESENT;

    }
}

