package org.alfresco.aos;

import static org.alfresco.utility.Utility.checkObjectIsInitialized;
import static org.alfresco.utility.report.log.Step.STEP;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Paths;
import java.util.List;

import org.alfresco.aos.dsl.AOSAssertion;
import org.alfresco.aos.dsl.AOSNetworkDrive;
import org.alfresco.aos.dsl.AOSUtil;
import org.alfresco.aos.dsl.JmxUtil;
import org.alfresco.utility.LogFactory;
import org.alfresco.utility.TasProperties;
import org.alfresco.utility.Utility;

import org.alfresco.utility.data.RandomData;
import org.alfresco.utility.dsl.DSLContentModelAction;
import org.alfresco.utility.dsl.DSLFile;
import org.alfresco.utility.dsl.DSLFolder;
import org.alfresco.utility.dsl.DSLProtocolWithNetworkDrive;
import org.alfresco.utility.exception.TestConfigurationException;
import org.alfresco.utility.model.ContentModel;
import org.alfresco.utility.model.FileModel;
import org.alfresco.utility.model.FileType;
import org.alfresco.utility.model.FolderModel;
import org.alfresco.utility.model.SiteModel;
import org.alfresco.utility.model.UserModel;
import org.apache.chemistry.opencmis.client.api.Document;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.lang3.SystemUtils;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.methods.RequestBuilder;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.xslf.usermodel.SlideLayout;
import org.apache.poi.xslf.usermodel.XMLSlideShow;
import org.apache.poi.xslf.usermodel.XSLFShape;
import org.apache.poi.xslf.usermodel.XSLFSlide;
import org.apache.poi.xslf.usermodel.XSLFSlideLayout;
import org.apache.poi.xslf.usermodel.XSLFSlideMaster;
import org.apache.poi.xslf.usermodel.XSLFTextShape;
import org.apache.poi.xssf.usermodel.XSSFCell;
import org.apache.poi.xssf.usermodel.XSSFRow;
import org.apache.poi.xssf.usermodel.XSSFSheet;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.apache.poi.xwpf.extractor.XWPFWordExtractor;
import org.apache.poi.xwpf.usermodel.BreakType;
import org.apache.poi.xwpf.usermodel.XWPFDocument;
import org.apache.poi.xwpf.usermodel.XWPFParagraph;
import org.apache.poi.xwpf.usermodel.XWPFRun;
import org.slf4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Service;

@Service
@Scope(value = "prototype")
public class AOSWrapper extends DSLProtocolWithNetworkDrive<AOSWrapper> implements DSLContentModelAction<AOSWrapper>, DSLFolder<AOSWrapper>, DSLFile<AOSWrapper>
{
    static Logger LOG = LogFactory.getLogger();

    @Autowired
    AOSProperties aosProperties;

    @Autowired
    TasProperties tasProperties;

    @Autowired
    AOSNetworkDrive aosNetworkDrive;

    @Autowired
    DataContentExtented dataContent;

    private FileOutputStream fileOut;
    private FileType fileType;
    private FileModel fileModel;

    private String content;
    private String aosLocation;
    public int status = HttpStatus.SC_INTERNAL_SERVER_ERROR;

    public enum Content { ADD, DELETE };

    public AOSNetworkDrive withNetworkDrive()
    {
        return aosNetworkDrive;
    }

    public AOSUtil withAosUtil()
    {
        return new AOSUtil(this);
    }

    @Override
    public AOSWrapper createFile(FileModel fileModel) throws Exception
    {
        content = RandomData.getRandomName("fileContent");
        switch (fileModel.getFileType())
        {
            case MSEXCEL:
            case MSEXCEL2007:
                for (int i = 0; i < 2; i++)
                {
                    XSSFWorkbook wb = new XSSFWorkbook();
                    XSSFSheet sheet = wb.createSheet();
                    sheet.createRow(1);
                    sheet.getRow(1).createCell(1);
                    sheet.getRow(1).getCell(1).setCellValue(fileModel.getContent());
                    LOG.info("Aos Location is {} and fileModel Name is {}", getAosLocation(), fileModel.getName());
                    fileOut = new FileOutputStream(new File(getAosLocation(), fileModel.getName()));
                    wb.write(fileOut);

                    fileOut.close();
                    wb.close();
                    setFileType(FileType.MSEXCEL);
                }
                break;

            case MSPOWERPOINT:
            case MSPOWERPOINT2007:
                for (int i = 0; i < 2; i++)
                {
                    XMLSlideShow ppt = new XMLSlideShow();
                    XSLFSlideMaster slideMaster = ppt.getSlideMasters().get(0);
                    XSLFSlideLayout titleLayout = slideMaster.getLayout(SlideLayout.TITLE);
                    XSLFSlide slide1 = ppt.createSlide(titleLayout);
                    XSLFTextShape title1 = slide1.getPlaceholder(0);
                    title1.setText(fileModel.getContent());

                    fileOut = new FileOutputStream(new File(getAosLocation(), fileModel.getName()));
                    ppt.write(fileOut);

                    fileOut.close();
                    ppt.close();
                    setFileType(FileType.MSPOWERPOINT);
                }
                break;

            case MSWORD:
            case MSWORD2007:
                for (int i = 0; i < 2; i++)
                {
                    XWPFDocument document= new XWPFDocument();
                    fileOut = new FileOutputStream(new File(getAosLocation(), fileModel.getName()));

                    XWPFParagraph paragraph = document.createParagraph();
                    XWPFRun run = paragraph.createRun();
                    run.setText(fileModel.getContent());
                    document.write(fileOut);

                    fileOut.close();
                    document.close();
                    setFileType(FileType.MSWORD);
                }
                break;

            default:
                throw new TestConfigurationException("Unknown Office file type" + fileModel.getFileType());
        }

        String location = getLastResourceWithoutPrefix();
        setLastResource(buildPath(location, fileModel.toFile().getName()));
        setAosLocation(getPrefixSpace() + getLastResource());

        setDefaultFileContent(fileModel.getContent());
        setFileModel(fileModel);

        fileModel.setProtocolLocation(getCurrentSpace() + fileModel.getName());
        fileModel.setCmisLocation(getLastResourceWithoutPrefix());
        fileModel.setNodeRef(contentService.getNodeRefByPath(getTestUser().getUsername(), getTestUser().getPassword(), getLastResourceWithoutPrefix()));

        LOG.info("Creating file: {}", getLastResource());
        dataContent.waitUntilContentIsCreatedInLinux(getLastResource());

        return this;
    }

    @Override
    public AOSWrapper createFolder(FolderModel folderModel) throws Exception
    {
        String folderLocation = buildPath(getLastResource(), folderModel.getName());
        aosNetworkDrive.inOSEnvironment().createFolder(folderLocation);

        setLastResource(folderLocation);
        setAosLocation(getPrefixSpace() + getLastResource());

        folderModel.setProtocolLocation(getCurrentSpace() + folderModel.getName());
        folderModel.setCmisLocation(getLastResourceWithoutPrefix());

        LOG.info("Creating Folder: {}", getLastResource());
        dataContent.waitUntilContentIsCreatedInLinux(getLastResource());

        return this;
    }

    @Override
    public List<FileModel> getFiles() throws Exception
    {
        // TODO Auto-generated method stub
        return null;
    }

    @Override
    public List<FolderModel> getFolders() throws Exception
    {
        // TODO Auto-generated method stub
        return null;
    }

    @Override
    public AOSWrapper rename(String newName) throws Exception
    {
        aosNetworkDrive.inOSEnvironment().renameContent(getLastResource(), newName).getPath();

        setLastResource(buildPath(getCurrentSpace(), newName));
        setAosLocation(getPrefixSpace() + getLastResource());

        dataContent.waitUntilContentIsCreatedInLinux(getLastResource());
        getLastContentModel().setProtocolLocation(getCurrentSpace() + fileModel.getName());
        getLastContentModel().setCmisLocation(getLastResourceWithoutPrefix());
        return this;
    }

    @Override
    public AOSWrapper update(String content) throws Exception
    {
        String resourcePath = aosNetworkDrive.inOSEnvironment().getLocalVolumePath() + getLastResource();
        updateContent(resourcePath, content, Content.ADD);

        dataContent.waitUntilContentIsCreatedInLinux(getLastResource());

        return this;
    }

    public AOSWrapper deleteContent(String content) throws Exception
    {
        String resourcePath = aosNetworkDrive.inOSEnvironment().getLocalVolumePath() + getLastResource();
        updateContent(resourcePath, content, Content.DELETE);

        dataContent.waitUntilContentIsCreatedInLinux(getLastResource());

        return this;
    }

    public String buildPath(String parent, String... paths)
    {
        return Utility.buildPath(parent, paths);
    }

    @Override
    public AOSWrapper delete() throws Exception
    {
        String resourcePath =  getLastResource();
        checkObjectIsInitialized(resourcePath, "resourcePath");
        aosNetworkDrive.inOSEnvironment().deleteContent(resourcePath);

        dataContent.waitUntilContentIsCreatedInLinux(getLastResource());

        return this;
    }

    @Override
    public AOSWrapper copyTo(ContentModel destination) throws Exception
    {
        File file = new File(getLastResourceWithoutPrefix());
        String fileName = file.getName();
        String fileDestination = buildPath(destination.getCmisLocation(), fileName);

        if (Files.isRegularFile(Paths.get(getAosLocation()), LinkOption.NOFOLLOW_LINKS))
        {
            aosNetworkDrive.inOSEnvironment().copyFile(getLastResourceWithoutPrefix(), fileDestination);
        }
        else
        {
            aosNetworkDrive.inOSEnvironment().copyFolder(getLastResourceWithoutPrefix(), fileDestination);
        }

        setLastResource(fileDestination);
        setAosLocation(getPrefixSpace() + getLastResource());

        dataContent.waitUntilContentIsCreatedInLinux(getLastResource());

        return this;
    }

    public AOSWrapper updateContent(String resourcePath, String content, Content contentAction) throws FileNotFoundException, IOException {
        if (FileType.fromName(resourcePath).equals(FileType.MSWORD) || FileType.fromName(resourcePath).equals(FileType.MSWORD2007))
        {
            FileInputStream file = new FileInputStream(new File(resourcePath));
            XWPFDocument document = new XWPFDocument(file);
            List<XWPFParagraph> paragraphs = document.getParagraphs();
            List<XWPFRun> runs =  paragraphs.get(0).getRuns();
            XWPFRun run = runs.get(0);
            if(contentAction.equals(Content.ADD))
            {
                run.setText(paragraphs.get(0).getText() + content, 0);
            }
            else
            {
                run.setText(content, 0);
            }
            document.write(new FileOutputStream(new File(resourcePath)));
            document.close();

            setFileType(FileType.MSWORD);
        }

        if (FileType.fromName(resourcePath).equals(FileType.MSEXCEL2007) || FileType.fromName(resourcePath).equals(FileType.MSEXCEL))
        {
            FileInputStream file = new FileInputStream(new File(resourcePath));
            XSSFWorkbook wb = new XSSFWorkbook(file);
            Sheet sheet = wb.getSheetAt(0);
            Row row = sheet.getRow(1);
            Cell cell = row.getCell(1);
            if(contentAction.equals(Content.ADD))
            {
                cell.setCellValue(getDefaultFileContent() + content);
            }
            else
            {
                cell.setCellValue(content);
            }
            FileOutputStream fileOut = new FileOutputStream(new File(resourcePath));
            wb.write(fileOut);
            fileOut.close();
            wb.close();

            setFileType(FileType.MSEXCEL);
        }

        if (FileType.fromName(resourcePath).equals(FileType.MSPOWERPOINT) || FileType.fromName(resourcePath).equals(FileType.MSPOWERPOINT2007))
        {
            XMLSlideShow ppt = new XMLSlideShow(new FileInputStream(resourcePath));
            XSLFTextShape title1 =  ppt.getSlides().get(0).getPlaceholder(0);
            if(contentAction.equals(Content.ADD))
            {
                title1.setText(getDefaultFileContent() + content);
            }
            else
            {
                title1.setText(content);
            }
            FileOutputStream fileOut = new FileOutputStream(new File(resourcePath));
            ppt.write(fileOut);
            fileOut.close();
            ppt.close();

            setFileType(FileType.MSPOWERPOINT);
        }

        dataContent.waitUntilContentIsCreatedInLinux(getLastResource());

        return this;
    }

    /**
     * This method will add a new page for Word Documents, a new sheet for Excel Documents and a new slide for PowerPoint Documents
     * @return
     * @throws IOException
     * @throws TestConfigurationException
     */
    public AOSWrapper addNewDocumentPage() throws IOException, TestConfigurationException {
        String resourcePath = aosNetworkDrive.inOSEnvironment().getLocalVolumePath() + getLastResource();
        FileInputStream file = new FileInputStream(new File(resourcePath));

        if (FileType.fromName(resourcePath).equals(FileType.MSWORD) || FileType.fromName(resourcePath).equals(FileType.MSWORD2007))
        {
            XWPFDocument document = new XWPFDocument(file);

            XWPFParagraph paragraph = document.getLastParagraph();
            XWPFRun run = paragraph.createRun();
            run.addBreak(BreakType.PAGE);
            run.setText(getFileModel().getContent());
            paragraph.setPageBreak(true);

            document.write(new FileOutputStream(new File(resourcePath)));
            document.close();
        }

        if (FileType.fromName(resourcePath).equals(FileType.MSEXCEL2007) || FileType.fromName(resourcePath).equals(FileType.MSEXCEL))
        {
            XSSFWorkbook wb = new XSSFWorkbook(file);
            Sheet newSheet = wb.createSheet("newSheet");
            newSheet.createRow(1);
            newSheet.getRow(1).createCell(1);
            newSheet.getRow(1).getCell(1).setCellValue(fileModel.getContent());
            FileOutputStream fileOut = new FileOutputStream(new File(resourcePath));
            wb.write(fileOut);
            fileOut.close();
            wb.close();
        }

        if (FileType.fromName(resourcePath).equals(FileType.MSPOWERPOINT) || FileType.fromName(resourcePath).equals(FileType.MSPOWERPOINT2007))
        {
            XMLSlideShow ppt = new XMLSlideShow(new FileInputStream(resourcePath));

            XSLFSlideMaster slideMaster = ppt.getSlideMasters().get(0);
            XSLFSlideLayout titleLayout = slideMaster.getLayout(SlideLayout.TITLE);
            XSLFSlide newSlide = ppt.createSlide(titleLayout);
            XSLFTextShape title1 = newSlide.getPlaceholder(0);
            title1.setText(fileModel.getContent());

            FileOutputStream fileOut = new FileOutputStream(new File(resourcePath));
            ppt.write(fileOut);
            fileOut.close();
            ppt.close();
        }

        return this;
    }

    @Override
    public AOSWrapper moveTo(ContentModel destination) throws Exception
    {
        String fileName = new File(getLastResourceWithoutPrefix()).getName();
        String fileDestination = buildPath(destination.getCmisLocation(), fileName);

        if (Files.isRegularFile(Paths.get(getAosLocation()), LinkOption.NOFOLLOW_LINKS))
        {
            aosNetworkDrive.inOSEnvironment().moveFile(getLastResourceWithoutPrefix(), fileDestination);
        }
        else
        {
            aosNetworkDrive.inOSEnvironment().moveFolder(getLastResourceWithoutPrefix(), fileDestination);
        }

        setLastResource(fileDestination);
        setAosLocation(getPrefixSpace() + getLastResource());

        dataContent.waitUntilContentIsCreatedInLinux(getLastResource());

        return this;
    }

    public AOSWrapper httpMoveTo(ContentModel destination) throws IOException
    {
        String sourceUrl = aosProperties.getAosLinuxNetwork() + getLastResource();
        String destinationUrl = aosProperties.getAosLinuxNetwork() + destination.getCmisLocation();
        String unhashedString = String.format("%s:%s", getCurrentUser().getUsername(), getCurrentUser().getPassword());
        String basicAuth = "Basic " + new String(new Base64().encode(unhashedString.getBytes()));

        CloseableHttpClient httpclient = HttpClients.createDefault();
        HttpUriRequest moveRequest = RequestBuilder
                .create("MOVE")
                .setUri(sourceUrl)
                .addHeader("Overwrite", "F")
                .addHeader("Destination", destinationUrl)
                .addHeader("Authorization", basicAuth)
                .build();
        CloseableHttpResponse response = httpclient.execute(moveRequest);
        status = response.getStatusLine().getStatusCode();
        return this;
    }

    @Override
    protected String getProtocolJMXConfigurationStatus() throws Exception
    {
        return withJMX().getAOSServerConfigurationStatus();
    }

    @Override
    public AOSWrapper authenticateUser(UserModel userModel) throws Exception
    {
        setTestUser(userModel);
        return this;
    }

    @Override
    public AOSWrapper disconnect() throws Exception
    {
        setTestUser(new UserModel());

        return this;
    }

    @Override
    public AOSWrapper usingSite(String siteId) throws Exception
    {
        STEP(String.format("AOS: Navigate to site '%s/documentLibrary/'", siteId));
        checkObjectIsInitialized(siteId, "SiteID");
        setCurrentSpace(buildSiteDocumentLibraryPath(siteId, ""));
        setAosLocation(buildPath(getPrefixSpace(), getLastResource()));
        setActionExecutedOnMappedDrive(false);
        return this;
    }

    @Override
    public AOSWrapper usingSite(SiteModel siteModel) throws Exception
    {
        checkObjectIsInitialized(siteModel, "SiteModel");
        usingSite(siteModel.getId());
        return this;
    }

    public AOSWrapper usingRoot() throws TestConfigurationException, Exception
    {
        setCurrentSpace(getRootPath());
        setAosLocation(buildPath(getPrefixSpace(), getRootPath()));

        return this;
    }

    @Override
    public AOSWrapper usingUserHome(String username) throws Exception
    {
        // TODO Auto-generated method stub
        return null;
    }

    @Override
    public AOSWrapper usingUserHome() throws Exception
    {
        // TODO Auto-generated method stub
        return null;
    }

    @Override
    public String getRootPath() throws TestConfigurationException
    {
        return String.format("/");
    }

    @Override
    public String getSitesPath() throws TestConfigurationException
    {
        return String.format("%s%s", getRootPath(), "Sites");
    }

    @Override
    public String getUserHomesPath() throws TestConfigurationException
    {
        return String.format("%s%s", getRootPath(), "User Homes");
    }

    @Override
    public String getDataDictionaryPath() throws TestConfigurationException
    {
        return String.format("%s%s", getRootPath(), "Data Dictionary");
    }

    @Override
    public String getPrefixSpace()
    {
        String prefixSpace = "";

        if (SystemUtils.IS_OS_WINDOWS)
        {
            prefixSpace = "Y:";
        }

        else if (SystemUtils.IS_OS_LINUX)
        {
            prefixSpace = "/mnt/alfrescoAos";
        }

        else if (SystemUtils.IS_OS_MAC)
        {
            prefixSpace = "/data/alfrescoAos";
        }

        return prefixSpace;
    }

    @Override
    public AOSWrapper usingResource(ContentModel model) throws Exception
    {
        setLastContentModel(model);
        STEP(String.format("AOS: Navigate to '%s'", model.getName()));
        setCurrentSpace(model.getCmisLocation());
        setAosLocation(buildPath(getPrefixSpace(), model.getCmisLocation()));

        return this;
    }

    /**
     * Assertion DSL. Call this method to see available assertions available {@link AOSAssertion}
     */
    @Override
    public AOSAssertion assertThat()
    {
        return new AOSAssertion(this);
    }

    /**
     * @return JMX DSL for this wrapper
     */
    public JmxUtil withJMX()
    {
        return new JmxUtil(this, jmxBuilder.getJmxClient());
    }

    @Override
    public AOSWrapper usingNetworkDrive() throws Exception
    {
        STEP(String.format("AOS: map a drive"));

        if (!aosNetworkDrive.inOSEnvironment().isNetworkDriveMounted())
        {
            aosNetworkDrive.inOSEnvironment().mount();
        }

        return this;
    }

    public AOSWrapper unmountNetworkDrive() throws Exception
    {
        STEP(String.format("AOS: unmount a drive"));

        if (aosNetworkDrive.inOSEnvironment().isNetworkDriveMounted())
        {
            aosNetworkDrive.inOSEnvironment().unount();
        }

        return this;
    }

    public String getContent(String filePath) throws TestConfigurationException, IOException
    {
        FileInputStream fileStream = new FileInputStream(filePath);
        String content = "";

        if (getFileType().equals(FileType.MSWORD) || getFileType().equals(FileType.MSWORD2007))
        {
            XWPFDocument document = new XWPFDocument(fileStream);
            XWPFWordExtractor wordExtractor = new XWPFWordExtractor(document);

            content = wordExtractor.getText().trim();
            wordExtractor.close();
        }

        if (getFileType().equals(FileType.MSEXCEL2007) || getFileType().equals(FileType.MSEXCEL))
        {
            XSSFWorkbook workbook = new XSSFWorkbook(fileStream);
            XSSFSheet worksheet = workbook.getSheetAt(0);
            XSSFRow row1 = worksheet.getRow(1);
            XSSFCell cell1 = row1.getCell(1);

            content = cell1.getStringCellValue();
            workbook.close();
        }

        if (getFileType().equals(FileType.MSPOWERPOINT) || getFileType().equals(FileType.MSPOWERPOINT2007))
        {
            XMLSlideShow powerPoint = new XMLSlideShow(fileStream);
            XSLFShape shape = powerPoint.getSlides().get(0).getShapes().get(0);

            if (shape instanceof XSLFTextShape)
            {
                XSLFTextShape textShape = (XSLFTextShape) shape;
                String text = textShape.getText();

                content = text;
            }

            powerPoint.close();
        }

        return content;
    }

    public String getDefaultFileContent()
    {
        return content;
    }

    public void setDefaultFileContent(String content)
    {
        this.content = content;
    }

    public void setFileModel(FileModel fileModel)
    {
        this.fileModel = fileModel;
    }

    public FileModel getFileModel()
    {
        return fileModel;
    }

    public FileType getFileType()
    {
        return fileType;
    }

    public void setFileType(FileType fileType)
    {
        this.fileType = fileType;
    }

    public String getAosLocation() {
        return aosLocation;
    }

    public void setAosLocation(String aosLocation) {
        LOG.info(String.format("Set AOS location to '%s'", aosLocation));
        this.aosLocation = aosLocation;
    }

    public AOSWrapper checkOutDocument() throws InterruptedException {
        Document document = dataContent.getCMISDocument(getLastResource());

        document.checkOut();

        return this;
    }

    public AOSWrapper cancelCheckOutDocument() throws InterruptedException {
        Document document = dataContent.getCMISDocument(getLastResource());

        document.cancelCheckOut();

        return this;
    }
}