/*
 * Copyright 2005-2015 Alfresco Software, Ltd.  All rights reserved.
 *
 * This file is part of a proprietary Alfresco module.
 *
 * License rights for this program are granted under the terms of the "Alfresco
 * Component License", which defines the permitted uses of the module.
 * License terms can be found in the file license.txt distributed with this module.
 */
package org.alfresco.officeservices.testclient.office;

import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.alfresco.officeservices.testclient.ServiceCommunicationException;
import org.alfresco.officeservices.testclient.ServiceResponseException;
import org.alfresco.officeservices.testclient.AoserviceClientMessageReceiver;
import org.alfresco.officeservices.testclient.AoservicesClient;
import org.alfresco.officeservices.testclient.fpse.FPSEResponseElement;
import org.alfresco.officeservices.testclient.util.URLEncoder;
import org.alfresco.officeservices.testclient.util.URLPathDecoder;
import org.apache.http.client.utils.URIUtils;

public abstract class AbstractOfficeClient implements AoserviceClientMessageReceiver
{

	protected static URLEncoder urlEncoder;

    static
    {
        urlEncoder = new URLEncoder();
        urlEncoder.addSafeCharacter('-');
        urlEncoder.addSafeCharacter('_');
        urlEncoder.addSafeCharacter('.');
        urlEncoder.addSafeCharacter('*');
        urlEncoder.addSafeCharacter('/');
    }

    protected AoservicesClient servicesClient = null;

    protected OfficeClientMessageReceiver messageReceiver = null;

    protected List<String> vtiInfHosts = new ArrayList<String>();
    
    public AbstractOfficeClient(String username, String password)
    {
        // create the Aoservices client
        servicesClient = new AoservicesClient(username,password);
        servicesClient.setMessageReceiver(this);
    }

    public void setMessageReceiver(OfficeClientMessageReceiver receiver)
    {
        messageReceiver = receiver;
    }

    protected void emitMessage(String msg)
    {
        if(messageReceiver != null)
        {
            messageReceiver.message(msg);
        }
    }

    public void message(String msg)
    {
        emitMessage("    "+msg);
    }

    protected String buildHostString(URI target) throws ServiceCommunicationException
    {
        try
        {
            return URIUtils.createURI(target.getScheme(), target.getHost(), target.getPort(), null, null, null).toString();
        }
        catch (URISyntaxException e)
        {
            throw new ServiceCommunicationException(e);
        }
    }

    protected List<String> fpseVersionHosts = new ArrayList<String>();

    protected void ensureVtiInf(URI target) throws ServiceCommunicationException, ServiceResponseException
    {
        if(vtiInfHosts.contains(buildHostString(target)))
        {
            return;
        }
        vtiInfHosts.add(buildHostString(target));
        emitMessage("Getting _vti_inf.html...");
        AoservicesClient.VTI_INF_VERSION viv;
        viv = servicesClient.getVtiInformation(target);
        emitMessage("Service information file is of version "+viv);
    }

    protected Map<String,String> fpseOpenedHosts = new HashMap<String,String>();

    protected void ensureFpseVersion(URI target) throws ServiceCommunicationException, ServiceResponseException
    {
        if(fpseVersionHosts.contains(buildHostString(target)))
        {
            return;
        }
        fpseVersionHosts.add(buildHostString(target));
        emitMessage("Getting FPSE version...");
        AoservicesClient.FPSEVersion fpseVersion = servicesClient.getFPSEVersion(target);
        emitMessage("FPSE version major="+fpseVersion.major+" minor="+fpseVersion.minor+" phase="+fpseVersion.phase+" incr="+fpseVersion.increase);
    }

    public AbstractOfficeClient()
    {
        super();
    }

    protected String ensureOpenService(URI target) throws ServiceCommunicationException, ServiceResponseException
    {
        String cachedUsername = fpseOpenedHosts.get(target.toString());
        if(cachedUsername != null)
        {
            return cachedUsername;
        }
        emitMessage("Opening FPSE service...");
        FPSEResponseElement openServiceElement = servicesClient.getFPSEOpenService(target);
        String username = openServiceElement.getSubElement("vti_username").getValue();
        if(!username.startsWith("SX|"))
        {
            throw new ServiceResponseException("invalid username format");
        }
        username = username.substring(3);
        username = username.replace("&#92;","\\");
        emitMessage("openen service for user "+username);
        fpseOpenedHosts.put(target.toString(),username);
        return username;
    }
    
    public String urlDecodePath(String in)
    {
        try
        {
            return URLPathDecoder.decode(in, "UTF8");
        }
        catch(UnsupportedEncodingException uee)
        {
            throw new RuntimeException("Missing UTF8 support", uee);
        }
    }
    
    public abstract void fileOpenDialog(URI target) throws ServiceCommunicationException, ServiceResponseException;

    public abstract void fileSaveDialog(URI target) throws ServiceCommunicationException, ServiceResponseException;
    
    public abstract OfficeFileHandle openFile(URI target) throws ServiceCommunicationException, ServiceResponseException;
    
    public abstract void refreshLock(OfficeFileHandle ofh) throws ServiceCommunicationException, ServiceResponseException;
    
    public abstract void closeFile(OfficeFileHandle ofh) throws ServiceCommunicationException, ServiceResponseException;
    
    public abstract void checkoutFile(OfficeFileHandle ofh) throws ServiceCommunicationException, ServiceResponseException;
    
    public abstract void checkinFile(OfficeFileHandle ofh, String comment, boolean keepEditing) throws ServiceCommunicationException, ServiceResponseException;

}