/*
 * Copyright 2005-2015 Alfresco Software, Ltd.  All rights reserved.
 *
 * This file is part of a proprietary Alfresco module.
 *
 * License rights for this program are granted under the terms of the "Alfresco
 * Component License", which defines the permitted uses of the module.
 * License terms can be found in the file license.txt distributed with this module.
 */
package org.alfresco.officeservices.testclient.office;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;

import org.alfresco.officeservices.testclient.ServiceCommunicationException;
import org.alfresco.officeservices.testclient.ServiceResponseException;
import org.alfresco.officeservices.testclient.AoservicesClient;
import org.alfresco.officeservices.testclient.AoservicesClient.FPSEGetDocsMetaInfo;
import org.alfresco.officeservices.testclient.AoservicesClient.FileStatus;
import org.alfresco.officeservices.testclient.AoservicesClient.OptionsHeaderValidation;
import org.alfresco.officeservices.testclient.fpse.FPSEResponseElement;
import org.apache.http.client.utils.URIUtils;

public class Office2007WindowsXPClient extends AbstractOfficeClient
{
    
    public Office2007WindowsXPClient(String username, String password)
    {
        super(username, password);
    }
    
    public void fileOpenDialog(URI target) throws ServiceCommunicationException, ServiceResponseException
    {
        fileDialog(target, AoservicesClient.DialogType.FileOpen);
    }
    
    public void fileSaveDialog(URI target) throws ServiceCommunicationException, ServiceResponseException
    {
        fileDialog(target, AoservicesClient.DialogType.FileSave);
    }
    
    public void fileDialog(URI target, AoservicesClient.DialogType dialogType) throws ServiceCommunicationException, ServiceResponseException
    {
        boolean repeated = false;
        AoservicesClient.FPSEUrlToWebUrl fpseUrlToWebUrl = null;
        while(true)
        {
            // OPTIONS requests
            emitMessage("Getting server options...");
            try
            {
                if(!servicesClient.options(URIUtils.createURI(target.getScheme(), target.getHost(), target.getPort(), null, null, null),OptionsHeaderValidation.NONE))
                {
                    throw new ServiceResponseException("OPTIONS request for server root failed");
                }
            }
            catch(URISyntaxException use)
            {
                throw new ServiceResponseException(use);
            }
            emitMessage("Server valid.");
            emitMessage("Getting target options...");
            if(!servicesClient.options(target,OptionsHeaderValidation.NONE))
            {
                throw new ServiceResponseException("OPTIONS request for target failed");
            }
            emitMessage("valid.");
            // _vti_inf.html
            ensureVtiInf(target);
            // FPSE version
            ensureFpseVersion(target);
            // url to web url
            emitMessage("Getting fileUrl and webUrl...");
            fpseUrlToWebUrl = servicesClient.getFPSEUrlToWebUrl(target);
            emitMessage("webUrl="+fpseUrlToWebUrl.webUrl+" fileUrl="+fpseUrlToWebUrl.fileUrl);
            // open service
            ensureOpenService(fpseUrlToWebUrl.fpseEndpoint);
            // test status of sub directories
            if(fpseUrlToWebUrl.fileUrl.length()>0)
            {
                emitMessage("Getting FPSE status of "+fpseUrlToWebUrl.fileUrl+"...");
                ArrayList<String> targets = new ArrayList<String>(1);
                targets.add(fpseUrlToWebUrl.fileUrl);
                FPSEGetDocsMetaInfo docsMetaInfo = servicesClient.getFPSEGetDocsMetaInfo(fpseUrlToWebUrl.fpseEndpoint,targets);
                FPSEResponseElement metaInfo = docsMetaInfo.urldirs.get(fpseUrlToWebUrl.fileUrl);
                if(metaInfo == null)
                {
                    emitMessage("Target invalid.");
                    if(repeated)
                    {
                        fpseUrlToWebUrl.fileUrl = "";
                        continue;
                    }
                    else
                    {
                        int pos = target.getPath().lastIndexOf('/');
                        if(pos == -1)
                        {
                            fpseUrlToWebUrl.fileUrl = "";
                            continue;
                        }
                        else
                        {
                            try
                            {
                                target = URIUtils.createURI(target.getScheme(), target.getHost(), target.getPort(), target.getPath().substring(0, pos-1), null, null);
                            }
                            catch(URISyntaxException use)
                            {
                                throw new ServiceResponseException(use);
                            }
                        }
                    }
                    repeated = true;
                    continue;
                }
                else
                {
                    emitMessage("Target is valid.");
                    break;
                }
            }
            break;
        }
        // test file dialog web form existence
        emitMessage("Testing Web Form existance...");
        boolean webFormExists = servicesClient.testWebFormExistence(fpseUrlToWebUrl.fpseEndpoint,fpseUrlToWebUrl.fileUrl,AoservicesClient.DialogType.FileOpen);
        if(webFormExists)
        {
            emitMessage("Web form exists.");
        }
        else
        {
            throw new ServiceResponseException("Web form for target does not exist. target:"+target);
        }
        // get file dialog web form
        emitMessage("Getting Web Form...");
        boolean webFormDelivered = servicesClient.getWebForm(fpseUrlToWebUrl.fpseEndpoint,fpseUrlToWebUrl.fileUrl,dialogType,null);
        if(webFormDelivered)
        {
            emitMessage("Web form received.");
        }
        else
        {
            throw new ServiceResponseException("Web form not received. target:"+target);
        }
    }
    
    public OfficeFileHandle openFile(URI target) throws ServiceCommunicationException, ServiceResponseException
    {
        // test existence
        emitMessage("Getting target options...");
        if(!servicesClient.options(target,OptionsHeaderValidation.ALL))
        {
            throw new ServiceResponseException("OPTIONS request for target failed");
        }
        emitMessage("valid.");
        // _vti_inf.html
        ensureVtiInf(target);
        // FPSE version
        ensureFpseVersion(target);
        // url to web url
        emitMessage("Getting fileUrl and webUrl...");
        AoservicesClient.FPSEUrlToWebUrl fpseUrlToWebUrl = servicesClient.getFPSEUrlToWebUrl(target);
        emitMessage("webUrl="+fpseUrlToWebUrl.webUrl+" fileUrl="+fpseUrlToWebUrl.fileUrl);
        // open service
        String username = ensureOpenService(fpseUrlToWebUrl.fpseEndpoint);
        // get meta info
        emitMessage("Getting FPSE status of "+target.toString()+"...");
        ArrayList<String> targets = new ArrayList<String>(1);
        targets.add(urlDecodePath(target.toString()));
        String x = target.getPath();
        String libraryPath = null;
        if(x.startsWith(fpseUrlToWebUrl.webUrl+"/"))
        {
            x = x.substring(fpseUrlToWebUrl.webUrl.length()+1);
            int pos = x.indexOf('/');
            if(pos > 0)
            {
                try
                {
                    libraryPath = fpseUrlToWebUrl.webUrl.substring(1) + "/" + x.substring(0, pos);
                    URI docLibUri = URIUtils.createURI(target.getScheme(), target.getHost(), target.getPort(), "/"+libraryPath, null, null);
                    targets.add(urlDecodePath(docLibUri.toString()));
                    emitMessage("...and FPSE status of "+docLibUri.toString()+"...");
                }
                catch(URISyntaxException use)
                {
                    throw new ServiceResponseException(use);
                }
            }
        }
        FPSEGetDocsMetaInfo docsMetaInfo = servicesClient.getFPSEGetDocsMetaInfo(fpseUrlToWebUrl.fpseEndpoint,targets);
        FileStatus status = docsMetaInfo.document_list.get(fpseUrlToWebUrl.fileUrl);
        emitMessage("Received file status "+status);
        if(libraryPath != null)
        {
            FPSEResponseElement LibraryDescription = docsMetaInfo.urldirs.get(libraryPath);
            if(LibraryDescription == null)
            {
                emitMessage("WARNING: Library description missing");
            }
        }
        // head to target
        emitMessage("Testing HTTP status of target "+target.toString()+"...");
        if(!servicesClient.doHead(target))
        {
            throw new ServiceResponseException("HEAD request for target failed");
        }
        emitMessage("HTTP status valid");
        // checkout document
        emitMessage("Checkout document...");
        AoservicesClient.FileStatus checkoutStatus = servicesClient.getFPSECheckoutDocument(fpseUrlToWebUrl.fpseEndpoint,fpseUrlToWebUrl.fileUrl,10);
        emitMessage("Received document status: "+checkoutStatus);
        // getDocsMetaInfo
        emitMessage("Getting FPSE status of "+fpseUrlToWebUrl.fileUrl+"...");
        targets = new ArrayList<String>(1);
        targets.add(fpseUrlToWebUrl.fileUrl);
        docsMetaInfo = servicesClient.getFPSEGetDocsMetaInfo(fpseUrlToWebUrl.fpseEndpoint,targets);
        status = docsMetaInfo.document_list.get(fpseUrlToWebUrl.fileUrl);
        emitMessage("Received status "+status);
        // uncheckout
        emitMessage("Uncheckout document...");
        status = servicesClient.getFPSEUncheckoutDocument(fpseUrlToWebUrl.fpseEndpoint,fpseUrlToWebUrl.fileUrl,true);
        emitMessage("Received document status: "+status);
        // get the document
        emitMessage("Getting document...");
        AoservicesClient.FileStatus documentStatus = servicesClient.getFPSEGetDocument(fpseUrlToWebUrl.fpseEndpoint,fpseUrlToWebUrl.fileUrl);
        emitMessage("Received document status: "+documentStatus);
        // head to target
        emitMessage("Testing HTTP status of target "+target.toString()+"...");
        if(!servicesClient.doHead(target))
        {
            throw new ServiceResponseException("HEAD request for target failed");
        }
        emitMessage("HTTP status valid");
        // create FileHandle
        OfficeFileHandle ofh = new OfficeFileHandle();
        ofh.target = target;
        ofh.fileUrl = fpseUrlToWebUrl.fileUrl;
        ofh.webUrl = fpseUrlToWebUrl.webUrl;
        ofh.fpseEndpoint = fpseUrlToWebUrl.fpseEndpoint;
        ofh.username = username;
        return ofh;
    }
    
    public void refreshLock(OfficeFileHandle ofh) throws ServiceCommunicationException, ServiceResponseException
    {
        // checkout
        emitMessage("Checkout document...");
        AoservicesClient.FileStatus status = servicesClient.getFPSECheckoutDocument(ofh.fpseEndpoint,ofh.fileUrl,10);
        emitMessage("Received document status: "+status);
    }
    
    public void closeFile(OfficeFileHandle ofh) throws ServiceCommunicationException, ServiceResponseException
    {
        // uncheckout
        emitMessage("Uncheckout document...");
        AoservicesClient.FileStatus status = servicesClient.getFPSEUncheckoutDocument(ofh.fpseEndpoint,ofh.fileUrl,true);
        emitMessage("Received document status: "+status);
    }
    
    public void checkoutFile(OfficeFileHandle ofh) throws ServiceCommunicationException, ServiceResponseException
    {
        // checkout
        emitMessage("Checkout document...");
        if(!servicesClient.soapCheckout(ofh.fpseEndpoint, ofh.target, false))
        {
            throw new ServiceResponseException("Checkout failed");
        }
        emitMessage("Checkout successfull");
        // getDocsMetaInfo
        emitMessage("Getting FPSE status of "+ofh.fileUrl+"...");
        ArrayList<String> targets = new ArrayList<String>(1);
        targets.add(ofh.fileUrl);
        FPSEGetDocsMetaInfo docsMetaInfo = servicesClient.getFPSEGetDocsMetaInfo(ofh.fpseEndpoint,targets);
        FileStatus status = docsMetaInfo.document_list.get(ofh.fileUrl);
        emitMessage("Received status "+status);
    }
    
    public void checkinFile(OfficeFileHandle ofh, String comment, boolean keepEditing) throws ServiceCommunicationException, ServiceResponseException
    {
        // checkin
        emitMessage("Checkin document...");
        if(!servicesClient.soapCheckin(ofh.fpseEndpoint, ofh.target, comment))
        {
            throw new ServiceResponseException("Checkin failed");
        }
        emitMessage("Checkin successfull");
        // uncheckout
        if(!keepEditing)
        {
            emitMessage("Uncheckout document...");
            FileStatus status = servicesClient.getFPSEUncheckoutDocument(ofh.fpseEndpoint,ofh.fileUrl,true);
            emitMessage("Received document status: "+status);
        }
        // get meta info
        emitMessage("Getting FPSE status of "+ofh.target.toString()+"...");
        ArrayList<String> targets = new ArrayList<String>(1);
        targets.add(urlDecodePath(ofh.target.toString()));
        String x = ofh.target.getPath();
        String libraryPath = null;
        if(x.startsWith(ofh.webUrl+"/"))
        {
            x = x.substring(ofh.webUrl.length()+1);
            int pos = x.indexOf('/');
            if(pos > 0)
            {
                try
                {
                    libraryPath = ofh.webUrl.substring(1) + "/" + x.substring(0, pos);
                    URI docLibUri = URIUtils.createURI(ofh.target.getScheme(), ofh.target.getHost(), ofh.target.getPort(), "/"+libraryPath, null, null);
                    targets.add(urlDecodePath(docLibUri.toString()));
                    emitMessage("...and FPSE status of "+docLibUri.toString()+"...");
                }
                catch(URISyntaxException use)
                {
                    throw new ServiceResponseException(use);
                }
            }
        }
        FPSEGetDocsMetaInfo docsMetaInfo = servicesClient.getFPSEGetDocsMetaInfo(ofh.fpseEndpoint,targets);
        FileStatus status = docsMetaInfo.document_list.get(ofh.fileUrl);
        emitMessage("Received status "+status);
        if(libraryPath != null)
        {
            FPSEResponseElement LibraryDescription = docsMetaInfo.urldirs.get(libraryPath);
            if(LibraryDescription == null)
            {
                emitMessage("WARNING: Library description missing");
            }
        }
    }

}
