/*
 * Copyright 2005-2015 Alfresco Software, Ltd.  All rights reserved.
 *
 * This file is part of a proprietary Alfresco module.
 *
 * License rights for this program are granted under the terms of the "Alfresco
 * Component License", which defines the permitted uses of the module.
 * License terms can be found in the file license.txt distributed with this module.
 */
package org.alfresco.officeservices.testclient;

import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringWriter;
import java.io.Writer;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.alfresco.officeservices.testclient.fpse.FPSERequest;
import org.alfresco.officeservices.testclient.fpse.FPSEResponse;
import org.alfresco.officeservices.testclient.fpse.FPSEResponseElement;
import org.alfresco.officeservices.testclient.util.URLEncoder;
import org.apache.commons.lang3.StringEscapeUtils;
import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.CookieStore;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpHead;
import org.apache.http.client.methods.HttpOptions;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.utils.URIUtils;
import org.apache.http.cookie.Cookie;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.BasicCookieStore;
import org.apache.http.impl.client.DefaultHttpClient;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;

public class AoservicesClient
{

    protected static URLEncoder urlEncoder;

    static
    {
        urlEncoder = new URLEncoder();
        urlEncoder.addSafeCharacter('-');
        urlEncoder.addSafeCharacter('_');
        urlEncoder.addSafeCharacter('.');
        urlEncoder.addSafeCharacter('*');
        urlEncoder.addSafeCharacter('/');
    }
    
    protected DefaultHttpClient httpClient = null;
    
    protected CookieStore cookieStore = null;
    
    protected AoserviceClientMessageReceiver messageReceiver = null;
    
    public AoservicesClient(String username, String password)
    {
        // create the HTTP client
        httpClient = new DefaultHttpClient();
        // set credentials
        if(username != null)
        {
            httpClient.getCredentialsProvider().setCredentials(AuthScope.ANY,new UsernamePasswordCredentials(username,password));
        }
        // create cookie store
        cookieStore = new BasicCookieStore();
        httpClient.setCookieStore(cookieStore);
    }
    
    public void addCookie(Cookie cookie)
    {
        cookieStore.addCookie(cookie);
    }
    
    public void setMessageReceiver(AoserviceClientMessageReceiver receiver)
    {
        messageReceiver = receiver;
    }
    
    protected void emitMessage(String msg)
    {
        if(messageReceiver != null)
        {
            messageReceiver.message(msg);
        }
    }
    
    public enum VTI_INF_VERSION { VER6, VER12, VER14, VER15 };
    
    public VTI_INF_VERSION getVtiInformation(URI target) throws ServiceCommunicationException, ServiceResponseException
    {
        URI vtiInfUri;
        HttpResponse serviceResponse;
        try
        {
            // build URI to /_vti_inf.html in server root
            vtiInfUri = URIUtils.createURI(target.getScheme(), target.getHost(), target.getPort(), "/_vti_inf.html", null, null);
            // perform get
            HttpGet vtiInfGet = new HttpGet(vtiInfUri);
            serviceResponse = httpClient.execute(vtiInfGet);
        }
        catch(Exception e)
        {
            throw new ServiceCommunicationException(e);
        }
        // check response status
        if(serviceResponse.getStatusLine().getStatusCode() != 200)
        {
            if(serviceResponse.getStatusLine().getStatusCode() == 404)
            {
                throw new ServiceResponseException("The service information file is missing. URI="+vtiInfUri);
            }
            else
            {
                throw new ServiceResponseException("Unknow error getting the service information file. StatusCode="+serviceResponse.getStatusLine().getStatusCode()+" ReasonPhrase="+serviceResponse.getStatusLine().getReasonPhrase()+" URI="+vtiInfUri);
            }
        }
        // test response entity
        HttpEntity vtiInfEntity = serviceResponse.getEntity();
        if(vtiInfEntity == null)
        {
            throw new ServiceResponseException("The request for the service information file does not return a Entity. URI="+vtiInfUri);
        }
        byte[] vtiInf;
        try
        {
            InputStream in = vtiInfEntity.getContent();
            try
            {
                vtiInf = streamToByteArray(in);
            }
            finally
            {
                in.close();
            }
        }
        catch(IOException ioe)
        {
            throw new ServiceCommunicationException(ioe);
        }
        // compare against references
        byte[] sampleVer6 = getResourceAsByteArray("org/alfresco/aoservices/testclient/resources/vtiinf.wss2");
        byte[] sampleVer12 = getResourceAsByteArray("org/alfresco/aoservices/testclient/resources/vtiinf.wss3");
        byte[] sampleVer14 = getResourceAsByteArray("org/alfresco/aoservices/testclient/resources/vtiinf.ver14");
        byte[] sampleVer15 = getResourceAsByteArray("org/alfresco/aoservices/testclient/resources/vtiinf.ver15");
        if(arrayCompare(vtiInf,sampleVer6))
        {
            return VTI_INF_VERSION.VER6;
        }
        if(arrayCompare(vtiInf,sampleVer12))
        {
            return VTI_INF_VERSION.VER12;
        }
        if(arrayCompare(vtiInf,sampleVer14))
        {
            return VTI_INF_VERSION.VER14;
        }
/*        if(arrayCompare(vtiInf,sampleVer15))
        {*/
            return VTI_INF_VERSION.VER15;
/*        }
        throw new ServiceResponseException("Invalid content of service information file. URI="+vtiInfUri);*/
    }
    
    public enum OptionsHeaderValidation { NONE, IMPORTANT, ALL };
    
    /**
     * Performs a OPTIONS request to the given target and returns true if and only if the resource exists.
     * 
     * @param target the URI to test
     * 
     * @return true if and only if the resource exists
     * 
     * @throws ServiceCommunicationException
     * @throws ServiceResponseException
     */
    public boolean options(URI target, OptionsHeaderValidation headerValidation) throws ServiceCommunicationException, ServiceResponseException
    {
        HttpResponse serviceResponse = null;
        int maxRepeat = 1;
        while(maxRepeat >= 0)
        {
            try
            {
                // perform get
                HttpOptions options = new HttpOptions(target);
                serviceResponse = httpClient.execute(options);
                serviceResponse.getEntity().getContent().close();
            }
            catch(Exception e)
            {
                throw new ServiceCommunicationException(e);
            }
            if(serviceResponse.getStatusLine().getStatusCode() == 302)
            {
                String location = getHeaderValue(serviceResponse, "Location", true);
                emitMessage("Redirected from "+target.toString()+" to "+location);
                if(!location.startsWith(target.toString()))
                {
                    throw new ServiceCommunicationException("Invalid 302 redirect");
                }
                try
                {
                    target = new URI(location);
                }
                catch (URISyntaxException e)
                {
                    throw new ServiceCommunicationException(e);
                }
                maxRepeat--;
                continue;
            }
            break;
        }
        // check for required header fields
        if(headerValidation != OptionsHeaderValidation.NONE)
        {
            checkForRequiredHeaders(serviceResponse,(headerValidation == OptionsHeaderValidation.ALL));
        }
        // check response status
        if(serviceResponse.getStatusLine().getStatusCode() == 200)
        {
            return true;
        }
        if(serviceResponse.getStatusLine().getStatusCode() == 404)
        {
            return false;
        }
        throw new ServiceResponseException("Invalid response getting options. StatusCode="+serviceResponse.getStatusLine().getStatusCode()+" ReasonPhrase="+serviceResponse.getStatusLine().getReasonPhrase()+" URI="+target);
    }
    
    /**
     * Performs a PROPFIND request to the given target and returns true if and only if the resource exists.
     * 
     * @param target the URI to test
     * 
     * @return true if and only if the resource exists
     * 
     * @throws ServiceCommunicationException
     * @throws ServiceResponseException
     */
    public boolean propfind(URI target) throws ServiceCommunicationException, ServiceResponseException
    {
        HttpResponse serviceResponse = null;
        int maxRepeat = 1;
        while(maxRepeat >= 0)
        {
            try
            {
                // perform get
                HttpPropfind propfind = new HttpPropfind(target);
                serviceResponse = httpClient.execute(propfind);
                serviceResponse.getEntity().getContent().close();
            }
            catch(Exception e)
            {
                throw new ServiceCommunicationException(e);
            }
            if(serviceResponse.getStatusLine().getStatusCode() == 302)
            {
                String location = getHeaderValue(serviceResponse, "Location", true);
                emitMessage("Redirected from "+target.toString()+" to "+location);
                if(!location.startsWith(target.toString()))
                {
                    throw new ServiceCommunicationException("Invalid 302 redirect");
                }
                try
                {
                    target = new URI(location);
                }
                catch (URISyntaxException e)
                {
                    throw new ServiceCommunicationException(e);
                }
                maxRepeat--;
                continue;
            }
            break;
        }
        // check for required header fields
        // there are no headers required for PROPFIND. checkForRequiredHeaders(serviceResponse,true);
        // check response status
        if(serviceResponse.getStatusLine().getStatusCode() == 207)
        {
            return true;
        }
        if(serviceResponse.getStatusLine().getStatusCode() == 404)
        {
            return false;
        }
        throw new ServiceResponseException("Invalid response getting properties. StatusCode="+serviceResponse.getStatusLine().getStatusCode()+" ReasonPhrase="+serviceResponse.getStatusLine().getReasonPhrase()+" URI="+target);
    }
    
    public String lock(URI target, String currentLockToken) throws ServiceCommunicationException, ServiceResponseException
    {
        HttpResponse serviceResponse = null;
        try
        {
            // perform get
            HttpLock lock = new HttpLock(target);
            if(currentLockToken != null)
            {
                lock.setHeader("If", "(<"+currentLockToken+">)");
            }
            serviceResponse = httpClient.execute(lock);
        }
        catch(Exception e)
        {
            throw new ServiceCommunicationException(e);
        }
        // check for required header fields
        checkForRequiredHeaders(serviceResponse,true);
        // check response status
        if(serviceResponse.getStatusLine().getStatusCode() != 200)
        {
            throw new ServiceResponseException("Invalid LOCK response. StatusCode="+serviceResponse.getStatusLine().getStatusCode()+" ReasonPhrase="+serviceResponse.getStatusLine().getReasonPhrase()+" URI="+target);
        }
        // get XML document from response
        HttpEntity responseEntity = serviceResponse.getEntity();
        if(responseEntity == null)
        {
            throw new ServiceResponseException("FPSE request does not return a Entity.");
        }
        // convert response to XML document
        Document xmlResponse = null;
        try
        {
            InputStream in = responseEntity.getContent();
            try
            {
                try
                {
                    xmlResponse = getDocumentFromStream(in);
                }
                catch (DocumentException e)
                {
                    throw new ServiceResponseException("Error reading soap response.");
                }
            }
            finally
            {
                in.close();
            }
        }
        catch(IOException ioe)
        {
            throw new ServiceCommunicationException(ioe);
        }
        // get result from XML document
        Element propElem = xmlResponse.getRootElement();
        Element lockdiscoveryElem = propElem.element("lockdiscovery");
        Element activelockElem = lockdiscoveryElem.element("activelock");
        Element locktokenElem = activelockElem.element("locktoken");
        Element hrefElem = locktokenElem.element("href");
        return hrefElem.getText();
    }
    
    public void unlock(URI target, String currentLockToken) throws ServiceCommunicationException, ServiceResponseException
    {
        HttpResponse serviceResponse = null;
        try
        {
            // perform get
            HttpUnlock unlock = new HttpUnlock(target);
            unlock.setHeader("Lock-Token", "<"+currentLockToken+">");
            serviceResponse = httpClient.execute(unlock);
        }
        catch(Exception e)
        {
            throw new ServiceCommunicationException(e);
        }
        // check for required header fields
        checkForRequiredHeaders(serviceResponse,true);
        // check response status
        if(serviceResponse.getStatusLine().getStatusCode() != 204)
        {
            throw new ServiceResponseException("Invalid UNLOCK response. StatusCode="+serviceResponse.getStatusLine().getStatusCode()+" ReasonPhrase="+serviceResponse.getStatusLine().getReasonPhrase()+" URI="+target);
        }
    }
    
    /**
     * Execute the given FPSE request and return the response.
     * 
     * @param request the request to be executed
     * 
     * @return the FPSE result
     * 
     * @throws ServiceCommunicationException
     * @throws ServiceResponseException
     */
    public FPSEResponse execute(FPSERequest request) throws ServiceCommunicationException, ServiceResponseException
    {
        HttpResponse serviceResponse;
        try
        {
            // perform get
            HttpPost fpsePost = new HttpPost(request.getServiceEndpoint());
            fpsePost.setEntity(request.getRequestEntity());
            fpsePost.setHeader("MIME-Version", "1.0");
            fpsePost.setHeader("User-Agent", "MSFrontPage/6.0");
            fpsePost.setHeader("Content-Type", "application/x-www-form-urlencoded");
            fpsePost.setHeader("X-Vermeer-Content-Type", "application/x-www-form-urlencoded");
            serviceResponse = httpClient.execute(fpsePost);
        }
        catch(Exception e)
        {
            throw new ServiceCommunicationException(e);
        }
        // check for required header fields
        checkForRequiredHeaders(serviceResponse,false);
        // check response status
        if(serviceResponse.getStatusLine().getStatusCode() != 200)
        {
            throw new ServiceResponseException("Invalid response to FPSE request. StatusCode="+serviceResponse.getStatusLine().getStatusCode()+" ReasonPhrase="+serviceResponse.getStatusLine().getReasonPhrase()+" URI="+request.getServiceEndpoint());
        }
        // get XML document from response
        HttpEntity responseEntity = serviceResponse.getEntity();
        if(responseEntity == null)
        {
            throw new ServiceResponseException("FPSE request does not return a Entity.");
        }
        try
        {
            InputStream in = responseEntity.getContent();
            try
            {
                FPSEResponse response = new FPSEResponse(in);
                FPSEResponseElement statusElement = response.getTopElement().getSubElement("status");
                if(statusElement != null)
                {
                    FPSEResponseElement msgElement = statusElement.getSubElement("msg");
                    String exceptionMsg = "Service returned error.";
                    if(msgElement != null)
                    {
                        exceptionMsg += " ("+msgElement.getValue()+")";
                    }
                    throw new ServiceResponseException(exceptionMsg);
                }
                return response;
            }
            finally
            {
                in.close();
            }
        }
        catch(IOException ioe)
        {
            throw new ServiceCommunicationException(ioe);
        }
    }
    
    public class FPSEVersion
    {
        public int major;
        public int minor;
        public int phase;
        public int increase;
    }
    
    public FPSEVersion getFPSEVersion(URI target) throws ServiceCommunicationException, ServiceResponseException
    {
        // build service URI
        URI vtiBinUri;
        try
        {
            // build URI to /_vti_bin FPSE service in server root
            vtiBinUri = URIUtils.createURI(target.getScheme(), target.getHost(), target.getPort(), "/_vti_bin/shtml.dll/_vti_rpc", null, null);
        }
        catch(Exception e)
        {
            throw new ServiceCommunicationException(e);
        }
        // perform request
        FPSERequest serverVersionRequest = new FPSERequest(vtiBinUri,"server version","6.0.2.8164");
        FPSEResponse serverVersionResponse = execute(serverVersionRequest);
        // parse response
        FPSEVersion ver = new FPSEVersion();
        FPSEResponseElement rootElement = serverVersionResponse.getTopElement();
        FPSEResponseElement sourceControlElement = getMandatoryElement(rootElement,"source control");
        if(!sourceControlElement.getValue().equals("1"))
        {
            throw new ServiceResponseException("invalid source control flag");
        }
        FPSEResponseElement serverVersionElement = getMandatoryElement(rootElement,"server version");
        FPSEResponseElement majorElement = getMandatoryElement(serverVersionElement,"major ver");
        FPSEResponseElement minorElement = getMandatoryElement(serverVersionElement,"minor ver");
        FPSEResponseElement phaseElement = getMandatoryElement(serverVersionElement,"phase ver");
        FPSEResponseElement verIncrElement = getMandatoryElement(serverVersionElement,"ver incr");
        try
        {
            ver.major = Integer.parseInt(majorElement.getValue());
            ver.minor = Integer.parseInt(minorElement.getValue());
            ver.phase = Integer.parseInt(phaseElement.getValue());
            ver.increase = Integer.parseInt(verIncrElement.getValue());
        }
        catch(Exception e)
        {
            throw new ServiceResponseException("error decoding server version");
        }
        return ver;
    }
    
    public class FPSEUrlToWebUrl
    {
        public String webUrl;
        public String fileUrl;
        public URI fpseEndpoint;
    }
    
    public FPSEUrlToWebUrl getFPSEUrlToWebUrl(URI target) throws ServiceCommunicationException, ServiceResponseException
    {
        // build service URI
        URI vtiBinUri;
        try
        {
            // build URI to /_vti_bin FPSE service in server root
            vtiBinUri = URIUtils.createURI(target.getScheme(), target.getHost(), target.getPort(), "/_vti_bin/shtml.dll/_vti_rpc", null, null);
        }
        catch(Exception e)
        {
            throw new ServiceCommunicationException(e);
        }
        // perform request
        FPSERequest serverVersionRequest = new FPSERequest(vtiBinUri,"url to web url","6.0.2.8164");
        serverVersionRequest.addParameter("url", target.getPath());
        serverVersionRequest.addParameter("flags", "0");
        FPSEResponse serverVersionResponse = execute(serverVersionRequest);
        // parse response
        FPSEUrlToWebUrl urlToWebUrl = new FPSEUrlToWebUrl();
        FPSEResponseElement rootElement = serverVersionResponse.getTopElement();
        urlToWebUrl.webUrl = StringEscapeUtils.unescapeHtml4(getMandatoryElement(rootElement,"webUrl").getValue());
        urlToWebUrl.fileUrl = StringEscapeUtils.unescapeHtml4(getMandatoryElement(rootElement,"fileUrl").getValue());
        if(urlToWebUrl.fileUrl.length()>0)
        {
            String reconstructed = urlToWebUrl.webUrl + "/" + urlToWebUrl.fileUrl;
            if(!reconstructed.equals(target.getPath()))
            {
                throw new ServiceResponseException("invalid urlTowebUrl split");
            }
        }
        else
        {
            String compareTo = target.getPath();
            if( (compareTo.length()>1) && (compareTo.charAt(compareTo.length()-1)=='/') )
            {
                compareTo = compareTo.substring(0, compareTo.length()-1);
            }
            if(!urlToWebUrl.webUrl.equals(compareTo))
            {
                throw new ServiceResponseException("invalid urlTowebUrl split");
            }
        }
        try
        {
            // build service URI
            urlToWebUrl.fpseEndpoint = URIUtils.createURI(target.getScheme(), target.getHost(), target.getPort(), urlToWebUrl.webUrl, null, null);
        }
        catch(Exception e)
        {
            throw new ServiceCommunicationException(e);
        }
        return urlToWebUrl;
    }
    
    public FPSEResponseElement getFPSEOpenService(URI serviceEndpoint) throws ServiceCommunicationException, ServiceResponseException
    {
        // build service URI
        URI authorUri;
        try
        {
            // build URI to /_vti_bin FPSE service in server root
            authorUri = URIUtils.createURI(serviceEndpoint.getScheme(), serviceEndpoint.getHost(), serviceEndpoint.getPort(), serviceEndpoint.getPath() + "/_vti_bin/_vti_aut/author.dll", null, null);
        }
        catch(Exception e)
        {
            throw new ServiceCommunicationException(e);
        }
        // perform request
        FPSERequest openServiceRequest = new FPSERequest(authorUri,"open service","6.0.2.8164");
        openServiceRequest.addParameter("name", serviceEndpoint.getPath());
        FPSEResponse openServiceResponse = execute(openServiceRequest);
        // parse response
        FPSEResponseElement rootElement = openServiceResponse.getTopElement();
        FPSEResponseElement serviceElement = getMandatoryElement(rootElement,"service");
        if(!getMandatoryElement(serviceElement, "service_name").getValue().equals(serviceEndpoint.getPath()))
        {
            throw new ServiceResponseException("invalid service name in open service response");
        }
        return getMandatoryElement(serviceElement, "meta_info");
    }
    
    public class FPSEGetDocsMetaInfo
    {
        public Map<String,FPSEResponseElement> urldirs = new HashMap<String,FPSEResponseElement>();
        public Map<String,FileStatus> document_list = new HashMap<String,FileStatus>();
    }
    
    public FPSEGetDocsMetaInfo getFPSEGetDocsMetaInfo(URI serviceEndpoint, List<String> targets) throws ServiceCommunicationException, ServiceResponseException
    {
        // build service URI
        URI authorUri;
        try
        {
            // build URI to /_vti_bin FPSE service in server root
            authorUri = URIUtils.createURI(serviceEndpoint.getScheme(), serviceEndpoint.getHost(), serviceEndpoint.getPort(), serviceEndpoint.getPath() + "/_vti_bin/_vti_aut/author.dll", null, null);
        }
        catch(Exception e)
        {
            throw new ServiceCommunicationException(e);
        }
        // perform request
        FPSERequest getDocsMetaInfoRequest = new FPSERequest(authorUri,"getDocsMetaInfo","6.0.2.8164");
        StringBuilder url_list = new StringBuilder();
        url_list.append('[');
        for(int i = 0; i < targets.size(); i++)
        {
            if(i>0)
            {
                url_list.append(';');
            }
            url_list.append(targets.get(i));
        }
        url_list.append(']');
        getDocsMetaInfoRequest.addParameter("url_list", url_list.toString());
        getDocsMetaInfoRequest.addParameter("listHiddenDocs", "false");
        getDocsMetaInfoRequest.addParameter("listLinkInfo", "false");
        FPSEResponse getDocsMetaInfoResponse = execute(getDocsMetaInfoRequest);
        // parse response
        FPSEGetDocsMetaInfo getDocsMetaInfo = new FPSEGetDocsMetaInfo();
        FPSEResponseElement rootElement = getDocsMetaInfoResponse.getTopElement();
        FPSEResponseElement urldirsElement = rootElement.getSubElement("urldirs");
        if(urldirsElement != null)
        {
            ArrayList<FPSEResponseElement> urldirsList = urldirsElement.getAnonymousLists();
            if(urldirsList != null)
            {
                for(FPSEResponseElement urldirElement : urldirsList)
                {
                    String url = getMandatoryElement(urldirElement, "url").getValue();
                    FPSEResponseElement metaInfo = getMandatoryElement(urldirElement, "meta_info");
                    getDocsMetaInfo.urldirs.put(url, metaInfo);
                }
            }
        }
        FPSEResponseElement document_listElement = rootElement.getSubElement("document_list");
        if(document_listElement != null)
        {
            ArrayList<FPSEResponseElement> document_listList = document_listElement.getAnonymousLists();
            if(document_listList != null)
            {
                for(FPSEResponseElement documentElement : document_listList)
                {
                    String document_name = getMandatoryElement(documentElement, "document_name").getValue();
                    FPSEResponseElement metaInfo = getMandatoryElement(documentElement, "meta_info");
                    getDocsMetaInfo.document_list.put(document_name, getFileStatusFromMetaInfo(metaInfo));
                }
            }
        }
        return getDocsMetaInfo;
    }
    
    public FileStatus getFPSEGetDocument(URI serviceEndpoint, String fileUrl) throws ServiceCommunicationException, ServiceResponseException
    {
        // build service URI
        URI authorUri;
        try
        {
            // build URI to /_vti_bin FPSE service in server root
            authorUri = URIUtils.createURI(serviceEndpoint.getScheme(), serviceEndpoint.getHost(), serviceEndpoint.getPort(), serviceEndpoint.getPath() + "/_vti_bin/_vti_aut/author.dll", null, null);
        }
        catch(Exception e)
        {
            throw new ServiceCommunicationException(e);
        }
        // perform request
        FPSERequest getDocumentRequest = new FPSERequest(authorUri,"get document","6.0.2.8164");
        getDocumentRequest.addParameter("service_name", serviceEndpoint.getPath());
        getDocumentRequest.addParameter("document_name", fileUrl);
        getDocumentRequest.addParameter("old_theme_html", "false");
        getDocumentRequest.addParameter("force", "false");
        getDocumentRequest.addParameter("get_option", "chkoutExclusive");
        getDocumentRequest.addParameter("doc_version", "");
        getDocumentRequest.addParameter("timeout", "10");
        FPSEResponse getDocumentResponse = execute(getDocumentRequest);
        // parse response
        FPSEResponseElement rootElement = getDocumentResponse.getTopElement();
        FPSEResponseElement documentElement = getMandatoryElement(rootElement,"document");
        return getFileStatusFromMetaInfo(getMandatoryElement(documentElement,"meta_info"));
    }
    
    public FileStatus getFPSECheckoutDocument(URI serviceEndpoint, String fileUrl, int timeout) throws ServiceCommunicationException, ServiceResponseException
    {
        // build service URI
        URI authorUri;
        try
        {
            // build URI to /_vti_bin FPSE service in server root
            authorUri = URIUtils.createURI(serviceEndpoint.getScheme(), serviceEndpoint.getHost(), serviceEndpoint.getPort(), serviceEndpoint.getPath() + "/_vti_bin/_vti_aut/author.dll", null, null);
        }
        catch(Exception e)
        {
            throw new ServiceCommunicationException(e);
        }
        // perform request
        FPSERequest checkoutDocumentRequest = new FPSERequest(authorUri,"checkout document","6.0.2.8164");
        checkoutDocumentRequest.addParameter("service_name", serviceEndpoint.getPath());
        checkoutDocumentRequest.addParameter("document_name", fileUrl);
        checkoutDocumentRequest.addParameter("force", (timeout==0)?"1":"2");
        checkoutDocumentRequest.addParameter("timeout", Integer.toString(timeout));
        FPSEResponse checkoutDocumentResponse = execute(checkoutDocumentRequest);
        // parse response
        FPSEResponseElement rootElement = checkoutDocumentResponse.getTopElement();
        return getFileStatusFromMetaInfo(getMandatoryElement(rootElement,"meta_info"));
    }
    
    public FileStatus getFPSEUncheckoutDocument(URI serviceEndpoint, String fileUrl, boolean rlsshortterm) throws ServiceCommunicationException, ServiceResponseException
    {
        // build service URI
        URI authorUri;
        try
        {
            // build URI to /_vti_bin FPSE service in server root
            authorUri = URIUtils.createURI(serviceEndpoint.getScheme(), serviceEndpoint.getHost(), serviceEndpoint.getPort(), serviceEndpoint.getPath() + "/_vti_bin/_vti_aut/author.dll", null, null);
        }
        catch(Exception e)
        {
            throw new ServiceCommunicationException(e);
        }
        // perform request
        FPSERequest uncheckoutDocumentRequest = new FPSERequest(authorUri,"uncheckout document","6.0.2.8164");
        uncheckoutDocumentRequest.addParameter("service_name", serviceEndpoint.getPath());
        uncheckoutDocumentRequest.addParameter("document_name", fileUrl);
        uncheckoutDocumentRequest.addParameter("force", "false");
        uncheckoutDocumentRequest.addParameter("rlsshortterm", rlsshortterm?"true":"false");
        FPSEResponse checkoutDocumentResponse = execute(uncheckoutDocumentRequest);
        // parse response
        FPSEResponseElement rootElement = checkoutDocumentResponse.getTopElement();
        return getFileStatusFromMetaInfo(getMandatoryElement(rootElement,"meta_info"));
    }
    
    public FileStatus getFPSECheckinDocument(URI serviceEndpoint, String fileUrl, String comment) throws ServiceCommunicationException, ServiceResponseException
    {
        // build service URI
        URI authorUri;
        try
        {
            // build URI to /_vti_bin FPSE service in server root
            authorUri = URIUtils.createURI(serviceEndpoint.getScheme(), serviceEndpoint.getHost(), serviceEndpoint.getPort(), serviceEndpoint.getPath() + "/_vti_bin/_vti_aut/author.dll", null, null);
        }
        catch(Exception e)
        {
            throw new ServiceCommunicationException(e);
        }
        // perform request
        FPSERequest checkinDocumentRequest = new FPSERequest(authorUri,"checkin document","6.0.2.8164");
        checkinDocumentRequest.addParameter("service_name", serviceEndpoint.getPath());
        checkinDocumentRequest.addParameter("document_name", fileUrl);
        checkinDocumentRequest.addParameter("comment", comment);
        checkinDocumentRequest.addParameter("keep_checked_out", "false");
        FPSEResponse checkoutDocumentResponse = execute(checkinDocumentRequest);
        // parse response
        FPSEResponseElement rootElement = checkoutDocumentResponse.getTopElement();
        return getFileStatusFromMetaInfo(getMandatoryElement(rootElement,"meta_info"));
    }
    
    public enum LockState { NONE, LOCKED, CHECKEDOUT };
    
    public class FileStatus
    {
        LockState lockState;
        String lockUser;
        FPSEResponseElement metaInfo;
        public String toString()
        {
            switch(lockState)
            {
            case NONE:
                return "not-locked";
            case LOCKED:
                return "short-term-locked by "+lockUser;
            case CHECKEDOUT:
                return "checked-out by "+lockUser;
            }
            return "--error--";
        }
    }
    
    public FileStatus getFileStatusFromMetaInfo(FPSEResponseElement metaInfo) throws ServiceResponseException
    {
        FileStatus fs = new FileStatus();
        fs.metaInfo = metaInfo;
        if(metaInfo.getSubElement("vti_sourcecontrolcheckedoutby") != null)
        {
            String username = metaInfo.getSubElement("vti_sourcecontrolcheckedoutby").getValue();
            if(!username.startsWith("SR|"))
            {
                throw new ServiceResponseException("invalid checkout username format");
            }
            username = username.substring(3);
            username = username.replace("&#92;","\\");
            fs.lockUser = username;
            if(metaInfo.getSubElement("vti_sourcecontrollockexpires") != null)
            {
                fs.lockState = LockState.LOCKED;
            }
            else
            {
                fs.lockState = LockState.CHECKEDOUT;
            }
        }
        else
        {
            fs.lockState = LockState.NONE;
            fs.lockUser = null;
        }
        return fs;
    }
    
    public boolean doHead(URI target) throws ServiceCommunicationException, ServiceResponseException
    {
        HttpResponse serviceResponse;
        try
        {
            // perform head
            HttpHead head = new HttpHead(target);
            serviceResponse = httpClient.execute(head);
        }
        catch(Exception e)
        {
            throw new ServiceCommunicationException(e);
        }
        // check for required header fields
        checkForRequiredHeaders(serviceResponse,true);
        // check response status
        if(serviceResponse.getStatusLine().getStatusCode() == 200)
        {
            return true;
        }
        if(serviceResponse.getStatusLine().getStatusCode() == 404)
        {
            return false;
        }
        throw new ServiceResponseException("Invalid response for web form existance. StatusCode="+serviceResponse.getStatusLine().getStatusCode()+" ReasonPhrase="+serviceResponse.getStatusLine().getReasonPhrase()+" target="+target);
    }
    
    public enum DialogType { FileOpen, FileSave, SaveForm };
    
    public boolean testWebFormExistence(URI serviceEndpoint, String location, DialogType type) throws ServiceCommunicationException, ServiceResponseException
    {
        URI webFormUri = null;
        try
        {
            // build URI to /_vti_bin FPSE service in server root
            webFormUri = URIUtils.createURI(serviceEndpoint.getScheme(), serviceEndpoint.getHost(), serviceEndpoint.getPort(), serviceEndpoint.getPath() + "/_vti_bin/owssvr.dll?location="+urlEncoder.encode(location)+"&dialogview="+type, null, null);
        }
        catch(Exception e)
        {
            throw new ServiceCommunicationException(e);
        }
        return doHead(webFormUri);
    }
    
    public boolean getWebForm(URI serviceEndpoint, String location, DialogType type, String filter) throws ServiceCommunicationException, ServiceResponseException
    {
        HttpResponse serviceResponse;
        try
        {
            // build URI to /_vti_bin FPSE service in server root
            if(filter == null)
            {
                filter = "*.*;";
            }
            URI webFormUri = URIUtils.createURI(serviceEndpoint.getScheme(), serviceEndpoint.getHost(), serviceEndpoint.getPort(), serviceEndpoint.getPath() + "/_vti_bin/owssvr.dll?location="+urlEncoder.encode(location)+"&dialogview="+type+"&FileDialogFilterValue="+filter, null, null);
            // perform get
            HttpGet webFormGet = new HttpGet(webFormUri);
            serviceResponse = httpClient.execute(webFormGet);
            serviceResponse.getEntity().getContent().close();
        }
        catch(Exception e)
        {
            throw new ServiceCommunicationException(e);
        }
        // check for required header fields
        checkForRequiredHeaders(serviceResponse,true);
        // check response status
        if(serviceResponse.getStatusLine().getStatusCode() == 200)
        {
            return true;
        }
        if(serviceResponse.getStatusLine().getStatusCode() == 404)
        {
            return false;
        }
        throw new ServiceResponseException("Invalid response for web form existance. StatusCode="+serviceResponse.getStatusLine().getStatusCode()+" ReasonPhrase="+serviceResponse.getStatusLine().getReasonPhrase()+" URI="+serviceEndpoint);
    }
    
    public boolean get(URI target) throws ServiceCommunicationException, ServiceResponseException
    {
        HttpResponse serviceResponse;
        try
        {
            // perform get
            HttpGet get = new HttpGet(target);
            serviceResponse = httpClient.execute(get);
            serviceResponse.getEntity().getContent().close();
        }
        catch(Exception e)
        {
            throw new ServiceCommunicationException(e);
        }
        // check for required header fields
        checkForRequiredHeaders(serviceResponse,true);
        // check response status
        if(serviceResponse.getStatusLine().getStatusCode() == 200)
        {
            return true;
        }
        if(serviceResponse.getStatusLine().getStatusCode() == 404)
        {
            return false;
        }
        throw new ServiceResponseException("Invalid response for web form existance. StatusCode="+serviceResponse.getStatusLine().getStatusCode()+" ReasonPhrase="+serviceResponse.getStatusLine().getReasonPhrase()+" URI="+target);
    }
    
    public boolean soapCheckout(URI serviceEndpoint, URI target, boolean checkoutToLocal) throws ServiceCommunicationException, ServiceResponseException
    {
        // build service URI
        URI listsUri;
        try
        {
            // build URI to /_vti_bin FPSE service in server root
            listsUri = URIUtils.createURI(serviceEndpoint.getScheme(), serviceEndpoint.getHost(), serviceEndpoint.getPort(), serviceEndpoint.getPath() + "/_vti_bin/lists.asmx", null, null);
        }
        catch(Exception e)
        {
            throw new ServiceCommunicationException(e);
        }
        // build soap request
        StringBuilder soapRequest = new StringBuilder();
        soapRequest.append("<?xml version=\"1.0\" encoding=\"utf-8\"?>");
        soapRequest.append("<soap:Envelope xmlns:soap=\"http://schemas.xmlsoap.org/soap/envelope/\">");
        soapRequest.append("<soap:Body>");
        soapRequest.append("<CheckOutFile xmlns=\"http://schemas.microsoft.com/sharepoint/soap/\">");
        soapRequest.append("<pageUrl>");
        soapRequest.append(target.toString());
        soapRequest.append("</pageUrl>");
        soapRequest.append("<checkoutToLocal>");
        soapRequest.append(checkoutToLocal ? "true" : "false");
        soapRequest.append("</checkoutToLocal>");
        soapRequest.append("<lastmodified>02 Jul 2011 17:25:58 -0000</lastmodified>");
        soapRequest.append("</CheckOutFile>");
        soapRequest.append("</soap:Body>");
        soapRequest.append("</soap:Envelope>");
        HttpResponse serviceResponse;
        try
        {
            // perform post
            HttpPost soapPost = new HttpPost(listsUri);
            soapPost.setEntity(new StringEntity(soapRequest.toString(),"text/xml","UTF-8"));
            soapPost.setHeader("Content-Type", "text/xml; charset=utf-8");
            soapPost.setHeader("SOAPAction", "http://schemas.microsoft.com/sharepoint/soap/CheckOutFile");
            soapPost.setHeader("User-Agent", "Microsoft Office/12.0 (Windows NT 5.1; Microsoft Office Word 12.0.4518; Pro)");
            soapPost.setHeader("X-Office-Version", "12.0.4518");
            serviceResponse = httpClient.execute(soapPost);
        }
        catch(Exception e)
        {
            throw new ServiceCommunicationException(e);
        }
        // check for required header fields
        checkForRequiredHeaders(serviceResponse,true);
        // check response status
        if(serviceResponse.getStatusLine().getStatusCode() != 200)
        {
            throw new ServiceResponseException("Invalid response to FPSE request. StatusCode="+serviceResponse.getStatusLine().getStatusCode()+" ReasonPhrase="+serviceResponse.getStatusLine().getReasonPhrase()+" URI="+serviceEndpoint);
        }
        // get XML document from response
        HttpEntity responseEntity = serviceResponse.getEntity();
        if(responseEntity == null)
        {
            throw new ServiceResponseException("FPSE request does not return a Entity.");
        }
        // convert response to XML document
        Document xmlResponse = null;
        try
        {
            InputStream in = responseEntity.getContent();
            try
            {
                try
                {
                    xmlResponse = getDocumentFromStream(in);
                }
                catch (DocumentException e)
                {
                    throw new ServiceResponseException("Error reading soap response.");
                }
            }
            finally
            {
                in.close();
            }
        }
        catch(IOException ioe)
        {
            throw new ServiceCommunicationException(ioe);
        }
        // get result from XML document
        Element soapEnvelope = xmlResponse.getRootElement();
        Element soapBody = soapEnvelope.element("Body");
        Element checkOutFileResponse = soapBody.element("CheckOutFileResponse");
        Element checkOutFileResult = checkOutFileResponse.element("CheckOutFileResult");
        String result = checkOutFileResult.getText();
        return result.equalsIgnoreCase("true");
    }
    
    public boolean soapCheckin(URI serviceEndpoint, URI target, String comment) throws ServiceCommunicationException, ServiceResponseException
    {
        // build service URI
        URI listsUri;
        try
        {
            // build URI to /_vti_bin FPSE service in server root
            listsUri = URIUtils.createURI(serviceEndpoint.getScheme(), serviceEndpoint.getHost(), serviceEndpoint.getPort(), serviceEndpoint.getPath() + "/_vti_bin/lists.asmx", null, null);
        }
        catch(Exception e)
        {
            throw new ServiceCommunicationException(e);
        }
        // build soap request
        StringBuilder soapRequest = new StringBuilder();
        soapRequest.append("<?xml version=\"1.0\" encoding=\"utf-8\"?>");
        soapRequest.append("<soap:Envelope xmlns:soap=\"http://schemas.xmlsoap.org/soap/envelope/\">");
        soapRequest.append("<soap:Body>");
        soapRequest.append("<CheckInFile xmlns=\"http://schemas.microsoft.com/sharepoint/soap/\">");
        soapRequest.append("<pageUrl>");
        soapRequest.append(target.toString());
        soapRequest.append("</pageUrl>");
        soapRequest.append("<comment>");
        soapRequest.append(comment);
        soapRequest.append("</comment>");
        soapRequest.append("<CheckinType>0</CheckinType>");
        soapRequest.append("</CheckInFile>");
        soapRequest.append("</soap:Body>");
        soapRequest.append("</soap:Envelope>");
        HttpResponse serviceResponse;
        try
        {
            // perform post
            HttpPost soapPost = new HttpPost(listsUri);
            soapPost.setEntity(new StringEntity(soapRequest.toString(),"text/xml","UTF-8"));
            soapPost.setHeader("Content-Type", "text/xml; charset=utf-8");
            soapPost.setHeader("SOAPAction", "http://schemas.microsoft.com/sharepoint/soap/CheckInFile");
            soapPost.setHeader("User-Agent", "Microsoft Office/12.0 (Windows NT 5.1; Microsoft Office Word 12.0.4518; Pro)");
            soapPost.setHeader("X-Office-Version", "12.0.4518");
            serviceResponse = httpClient.execute(soapPost);
        }
        catch(Exception e)
        {
            throw new ServiceCommunicationException(e);
        }
        // check for required header fields
        checkForRequiredHeaders(serviceResponse,true);
        // check response status
        if(serviceResponse.getStatusLine().getStatusCode() != 200)
        {
            throw new ServiceResponseException("Invalid response to FPSE request. StatusCode="+serviceResponse.getStatusLine().getStatusCode()+" ReasonPhrase="+serviceResponse.getStatusLine().getReasonPhrase()+" URI="+serviceEndpoint);
        }
        // get XML document from response
        HttpEntity responseEntity = serviceResponse.getEntity();
        if(responseEntity == null)
        {
            throw new ServiceResponseException("FPSE request does not return a Entity.");
        }
        // convert response to XML document
        Document xmlResponse = null;
        try
        {
            InputStream in = responseEntity.getContent();
            try
            {
                try
                {
                    xmlResponse = getDocumentFromStream(in);
                }
                catch (DocumentException e)
                {
                    throw new ServiceResponseException("Error reading soap response.");
                }
            }
            finally
            {
                in.close();
            }
        }
        catch(IOException ioe)
        {
            throw new ServiceCommunicationException(ioe);
        }
        // get result from XML document
        Element soapEnvelope = xmlResponse.getRootElement();
        Element soapBody = soapEnvelope.element("Body");
        Element checkInFileResponse = soapBody.element("CheckInFileResponse");
        Element checkInFileResult = checkInFileResponse.element("CheckInFileResult");
        String result = checkInFileResult.getText();
        return result.equalsIgnoreCase("true");
    }
    
    protected FPSEResponseElement getMandatoryElement(FPSEResponseElement elem, String name) throws ServiceResponseException
    {
        FPSEResponseElement result = elem.getSubElement(name);
        if(result == null)
        {
            throw new ServiceResponseException("sub element '"+name+"' missing");
        }
        return result;
    }
    
    /**
     * Checks the given HttpResponse for a set of required headers. Throws ServiceResponseException if headers are missing or invalid.
     * 
     * @param serviceResponse the HttpResponse to test
     * 
     * @throws ServiceResponseException if headers are missing or invalid
     */
    protected void checkForRequiredHeaders(HttpResponse serviceResponse, boolean strict) throws ServiceResponseException
    {
        // check header "MicrosoftSharePointTeamServices"
        String mspts = getHeaderValue(serviceResponse,"MicrosoftSharePointTeamServices",strict);
        if(! (mspts.startsWith("6.0.2.") || mspts.startsWith("12.0.0.") || mspts.startsWith("14.0.0.") || mspts.startsWith("15.0.0.")))
        {
            throw new ServiceResponseException("Invalid/Unexpected value for HTTP header MicrosoftSharePointTeamServices: "+mspts);
        }
        // check header "MS-Author-Via"
        String msav = getHeaderValue(serviceResponse,"MS-Author-Via",strict);
        if(!msav.equals("MS-FP/4.0,DAV"))
        {
            if(strict)
            {
                throw new ServiceResponseException("Invalid/Unexpected value for HTTP header MS-Author-Via");
            }
            else
            {
                emitMessage("WARNING: Invalid/Unexpected value for HTTP header MS-Author-Via");
            }
        }
        // check header "MicrosoftOfficeWebServer"
        String msows = getHeaderValue(serviceResponse,"MicrosoftOfficeWebServer",strict);
        if(!msows.equals("5.0_Collab"))
        {
            if(strict)
            {
                throw new ServiceResponseException("Invalid/Unexpected value for HTTP header MicrosoftOfficeWebServer");
            }
            else
            {
                emitMessage("WARNING: Invalid/Unexpected value for HTTP header MicrosoftOfficeWebServer");
            }
        }
        // check header "DocumentManagementServer"
        getHeaderValue(serviceResponse,"DocumentManagementServer",strict);
        // check header "DAV"
        String dav = getHeaderValue(serviceResponse,"DAV",strict);
        if(!dav.equals("1,2"))
        {
            if(strict)
            {
                throw new ServiceResponseException("Invalid/Unexpected value for HTTP header DAV");
            }
            else
            {
                emitMessage("WARNING: Invalid/Unexpected value for HTTP header DAV");
            }
        }
    }
    
    /**
     * Tries to get value from the last header with the given name. Throws ServiceResponseException if header is not present.
     * 
     * @param serviceResponse the response to get the header from
     * @param headerName the name of the header
     * 
     * @return the value from the last header with the given name
     * 
     * @throws ServiceResponseException if header is not present
     */
    protected static String getHeaderValue(HttpResponse serviceResponse, String headerName, boolean strict) throws ServiceResponseException
    {
        Header hdr = serviceResponse.getLastHeader(headerName);
        if( (hdr == null) || (hdr.getValue() == null) )
        {
            if(strict)
            {
                throw new ServiceResponseException("Required header is missing: "+headerName);
            }
            else
            {
                return "";
            }
        }
        return hdr.getValue();
    }
    
    /**
     * Compares two byte arrays and returns true if and only if both are identical
     * 
     * @param a1 the first array
     * @param a2 the second array
     * 
     * @return true if and only if both arrays are identical
     */
    protected static boolean arrayCompare(byte[] a1, byte a2[])
    {
        if(a1.length != a2.length)
        {
            return false;
        }
        for(int i = 0; i < a1.length; i++)
        {
            if(a1[i] != a2[i])
            {
                return false;
            }
        }
        return true;
    }
    
    /**
     * Reads the contents in the input stream and returns it as byte array.
     * 
     * @param in The input stream containing the data
     * 
     * @return the data from the input stream as byte array
     * 
     * @throws IOException
     */
    protected static byte[] streamToByteArray(InputStream in) throws IOException
    {
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        byte[] buffer = new byte[1024];
        int read = -1;
        while( (read = in.read(buffer)) >= 0)
        {
            bos.write(buffer, 0, read);
        }
        return bos.toByteArray();
    }
    
    /**
     * Returns a resource in the java ClassPath as byte array.
     * 
     * @param resource name of resource to load
     * 
     * @return the byte array
     */
    public static byte[] getResourceAsByteArray(String resource)
    {
        InputStream in = getResourceAsStream(resource);
        try
        {
            try
            {
                return streamToByteArray(in);
            }
            finally
            {
                in.close();
            }
        }
        catch(IOException ioe)
        {
            throw new RuntimeException("Error reading resource",ioe);
        }
    }

    /**
     * Returns a resource in the java ClassPath as stream.
     * 
     * @param resource name of resource to load
     * 
     * @return the stream
     */
    public static InputStream getResourceAsStream(String resource)
    {
        String stripped = resource.startsWith("/") ? resource.substring(1) : resource;
        InputStream stream = null;
        ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
        if(classLoader!=null)
        {
            stream = classLoader.getResourceAsStream(stripped);
        }
        if(stream == null)
        {
            stream = AoservicesClient.class.getResourceAsStream(resource);
        }
        if(stream == null)
        {
            stream = AoservicesClient.class.getClassLoader().getResourceAsStream(stripped);
        }
        if(stream == null)
        {
            throw new RuntimeException("Can not read resource " + resource);
        }
        return stream;
    }

    public Document getDocumentFromStream(InputStream is) throws IOException, DocumentException
    {
        Writer writer = new StringWriter();
        char[] buffer = new char[1024];
        try
        {
            Reader reader = new BufferedReader(new InputStreamReader(is, "UTF-8"));
            int n;
            while ((n = reader.read(buffer)) != -1)
            {
                writer.write(buffer, 0, n);
            }
        }
        finally
        {
            is.close();
        }
        return DocumentHelper.parseText(writer.toString());
    }
}
