/*
 * Decompiled with CFR 0.152.
 */
package org.alfresco.event.outbox;

import java.sql.Connection;
import java.sql.SQLException;
import java.time.Duration;
import java.time.Instant;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicBoolean;
import org.alfresco.event.outbox.ConnectionSource;
import org.alfresco.event.outbox.Event;
import org.alfresco.event.outbox.EventConsumer;
import org.alfresco.event.outbox.EventTable;
import org.alfresco.event.outbox.EventTableStats;
import org.alfresco.event.outbox.InternalEvent;
import org.alfresco.event.outbox.Metrics;
import org.alfresco.event.outbox.MetricsAdapter;
import org.alfresco.event.outbox.Outbox;
import org.alfresco.event.outbox.OutboxConfig;
import org.alfresco.event.outbox.OutboxException;
import org.alfresco.event.outbox.OutboxWorker;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class EventTableOutbox
implements Outbox {
    private static final Logger LOGGER = LoggerFactory.getLogger(EventTableOutbox.class);
    private final EventTable eventTable;
    private final ConnectionSource connectionSource;
    private final MetricsAdapter metricsAdapter;
    private final AtomicBoolean isSchemaUpToDate = new AtomicBoolean();
    private final OutboxWorker worker;
    private final OutboxConfig config;

    public EventTableOutbox(EventTable eventTable, OutboxWorker outboxWorker, ConnectionSource connectionSource, MetricsAdapter metricsAdapter, OutboxConfig config) {
        this.eventTable = Objects.requireNonNull(eventTable);
        this.worker = Objects.requireNonNull(outboxWorker);
        this.connectionSource = Objects.requireNonNull(connectionSource);
        this.metricsAdapter = Objects.requireNonNull(metricsAdapter);
        this.config = Objects.requireNonNull(config);
    }

    @Override
    public void start() {
        this.ensureSchemaIsUpToDate();
        this.worker.start();
    }

    @Override
    public void shutdown() {
        this.worker.stop();
    }

    @Override
    public EventConsumer getInboundEventConsumer() {
        return this::storeEvent;
    }

    EventTableStats getStats() {
        return this.worker.getStats();
    }

    private void storeEvent(Event event) {
        this.worker.ensureWorkerIsInDesiredState();
        this.ensureSchemaIsUpToDate();
        InternalEvent eventToStore = event.toInternalEvent();
        Instant start = Instant.now();
        boolean eventStored = false;
        try {
            if (this.connectionSource.isTxBoundConnectionAvailable()) {
                LOGGER.debug("Using Transaction bound connection for storing event.");
                this.storeEvent(this.connectionSource.getTxBoundConnection(), eventToStore);
            } else {
                LOGGER.debug("No Transaction bound connection available. New connection will be obtained.");
                try (Connection connection = this.connectionSource.getNonTxBoundConnection();){
                    this.storeEvent(connection, eventToStore);
                }
            }
            eventStored = true;
            this.metricsAdapter.report(eventStored ? Metrics.EVENT_STORED_SUCCESS : Metrics.EVENT_STORED_FAILURE, Duration.between(start, Instant.now()));
        }
        catch (SQLException e) {
            try {
                LOGGER.debug("Failed to insert {}", (Object)eventToStore, (Object)e);
                throw new OutboxException("Failed to insert `%s`.".formatted(eventToStore), e);
            }
            catch (Throwable throwable) {
                this.metricsAdapter.report(eventStored ? Metrics.EVENT_STORED_SUCCESS : Metrics.EVENT_STORED_FAILURE, Duration.between(start, Instant.now()));
                throw throwable;
            }
        }
    }

    private void storeEvent(Connection connection, InternalEvent eventToStore) throws SQLException {
        if (!this.eventTable.insertEvent(connection, eventToStore)) {
            LOGGER.debug("Failed to insert {}", (Object)eventToStore);
            throw new OutboxException("Failed to insert `%s`.".formatted(eventToStore));
        }
    }

    private void ensureSchemaIsUpToDate() {
        if (this.isSchemaUpToDate.get()) {
            return;
        }
        int remainingAttempts = this.config.getSchemaCreationMaxAttempts();
        while (remainingAttempts-- > 0) {
            try {
                try (Connection connection = this.connectionSource.getNonTxBoundConnection();){
                    if (this.eventTable.isSchemaUpToDate(connection)) {
                        LOGGER.info("Schema is up to date.");
                        this.isSchemaUpToDate.set(true);
                        return;
                    }
                }
                LOGGER.info("Trying to make schema up to date.");
                connection = this.connectionSource.getNonTxBoundConnection();
                try {
                    this.eventTable.tryToMakeSchemaUpToDate(connection);
                    LOGGER.info("Schema has been updated.");
                }
                finally {
                    if (connection == null) continue;
                    connection.close();
                }
            }
            catch (SQLException e) {
                if (remainingAttempts > 0) {
                    LOGGER.warn("Failed make schema up to date. Remaining retry attempts #{}.", (Object)remainingAttempts, (Object)e);
                    continue;
                }
                LOGGER.warn("Failed make schema up to date. No more attempts will be taken.", (Throwable)e);
                throw new OutboxException("Failed make schema up to date.", e);
            }
        }
        throw new OutboxException("Failed to make schema up to date.");
    }
}

